/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

import java.lang.instrument.ClassDefinition;
import java.net.URL;
import java.security.ProtectionDomain;
import java.util.Enumeration;

import org.zeroturnaround.javarebel.support.ScanHelper;

/**
 * <p>Provides hooks for integrating JRebel with custom containers, frameworks and classloaders.
 * To use it acquire an instance from {@link IntegrationFactory}</p>
 * 
 * <p>A typical (classloader) integration will look like this:</p>
 * 
 * <pre>
 * class MyClassLoader extends URLClassLoader {
 *   public MyClassLoader() {
 *     ...
 *     IntegrationFactory.getInstance().registerClassLoader(
 *      this,
 *      new FindResourceClassResourceSource(this)));
 *   }
 *   ...
 *   public Class findClass(String classname) {
 *     synchronized(classloader) {
 *       Class result =
 *         classloader.findLoadedClass(classname);
 *       if (result != null)
 *         return result;
 *         
 *       result = 
 *         IntegrationFactory.getInstance()
 *           .findReloadableClass(this, classname);
 *       if (result != null)
 *         return result;
 *     }
 *      ...
 *   }   
 * }
 * </pre>
 * 
 * <p>You can use the {@link #addIntegrationProcessor(String, ClassBytecodeProcessor)} and 
 * {@link #addIntegrationProcessor(String[], ClassBytecodeProcessor)} instrument specific 
 * classes to enhance their behaviour with JRebel.</p>
 * 
 * @author Jevgeni Kabanov
 * 
 * @see IntegrationFactory
 * @see ServletIntegration
 * @see Configuration
 * @see Reloader
 * @see Logger
 * @see Plugin
 */
public interface Integration {
  /**
   * Registers a <code>ClassLoader</code> with JRebel associating with it
   * the class-to-file resolution strategy provided by {@link ClassResourceSource}. It
   * is necessary to do this before any calls to {@link #findReloadableClass(ClassLoader, String)},
   * the best place is usually the constructor.
   * 
   * @param cl The custom class loader being integrated.
   * @param cfs The class-to-file resolution strategy source. 
   * 
   * @see ClassResourceSource
   */
  public void registerClassLoader(ClassLoader cl, ClassResourceSource cfs);

  /**
   * Reinitializes the <code>ClassLoader</code> with JRebel.
   * <p>
   * This should be called if the class path of the given <code>ClassLoader</code> was updated.
   * 
   * @param cl The custom class loader being integrated.
   */
  public void reinitializeClassLoader(ClassLoader cl);
  
  /**
   * Unregisters the <code>ClassLoader</code> from JRebel.
   * It is recommended to do this in case the corresponding <code>ClassLoader</code> is destroyed
   * (e.g. the web application is undeployed).
   * 
   * @param cl The custom class loader being integrated.
   */
  public void unregisterClassLoader(ClassLoader cl);

  /**
   * Says if this <code>ClassLoader</code> is registerd with JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @return true if this <code>ClassLoader</code> is registerd with JRebel.
   */
  public boolean isRegisteredClassLoader(ClassLoader cl);

  /**
   * Returns the additional class path of the given <code>ClassLoader</code>
   * managed by JRebel.
   * <p>
   * Same as <code>getRebelURLs(cl, false)</code>.
   * 
   * @param cl The custom class loader being integrated.
   * @return additional class path of the given <code>ClassLoader</code>
   *   or <code>null</code> if nothing found.
   */
  public URL[] getRebelURLs(ClassLoader cl);
  
  /**
   * Returns the additional class path of the given <code>ClassLoader</code>
   * managed by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param onlyFullMatch whether to only return locations with no custom patterns.
   * @return additional class path of the given <code>ClassLoader</code>
   *    or <code>null</code> if nothing found.
   */
  public URL[] getRebelURLs(ClassLoader cl, boolean onlyFullMatch);
  
  /**
   * Returns the additional class path of the given <code>ClassLoader</code>
   * managed by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @return additional class path of the given <code>ClassLoader</code>
   *   or <code>null</code> if nothing found.
   */
  public RebelSource[] getRebelSources(ClassLoader cl);
  
  /**
   * Returns helper object to perform class path scanning in the given
   * <code>ClassLoader</code>.
   * NOTE: this api is experimental and may be removed or changed in future
   * versions.
   * 
   * @param cl The class loader used to find resources to scan.
   * @param name name of resource to start scanning from.
   * @return helper for class path scanning or <code>null</code> if not available.
   */
  public ScanHelper getScanHelper(ClassLoader cl, String name);
    
  /**
   * Resolves classes managed by JRebel. These and only these classes
   * will be reloadable.
   * 
   * Returns <b>null</b> if class couldn't be resolved
   * and the non-JRebel class loading should be used as fail-over.
   * 
   * Throws {@link ClassNotFoundException} if class does not exist
   * and the non-JRebel class loading should be skipped.
   * 
   * @param cl The custom class loader being integrated.
   * @param className The public class name.
   * @return Resolved class or <b>null</b>.
   *
   * @throws ClassNotFoundException if class does not exist.
   * 
   * @see #defineReloadableClass(ClassLoader, String, Resource, ProtectionDomain, boolean)
   */
  public Class<?> findReloadableClass(ClassLoader cl, String className) throws ClassNotFoundException;

 /**
  * (Re)defines a class managed by JRebel.
  * 
  * When the class is reloaded, the Resource provided here is used to get the new bytes.
  * 
  * This can be used to re-generate the bytes for a proxy class after a reload.
  * 
  * Returns <b>null</b> if this class loader is not registered with JRebel.
  * 
  * @param cl The custom class loader being integrated.
  * @param className The public class name.
  * @param mutableResource bytes for the class.
  * @param protectionDomain The ProtectionDomain of the class (may be <code>null</code>).
  * @param processedByLoader Whether given bytes should be processed by class loader in the same way
  *   as a normally loaded classes. Set to <code>false</code> when class is supposed to be define similarly to
  *   reflectively calling <code>ClassLoader.defineClass</code> that bypasses logic inside defining loader.
  * @return (Re)defined class or <code>null</code>.
  * 
  * @since 18.2.5
  */
 public Class<?> defineReloadableClass(ClassLoader cl, String className, Resource mutableResource, ProtectionDomain protectionDomain,
     boolean processedByLoader);

  /**
   * (Re)defines a class managed by JRebel.
   * <p>
   * This method may be called for loading as well as reloading the class.
   * </p>
   * <p>
   * This method is intended to replace defineClass, e.g. used from the same context as defineClass is currently used.
   * So in normal java ClassLoader the caller of defineClass should already be synchronized and we should not do it inside defineReloadableClass in order to avoid creating extra deadlocks.
   * </p>
   * <p>
   * The defined class will only be reloaded if this method is called again.
   * Even if the class was first loaded by {@link #findReloadableClass(ClassLoader, String)}
   * the resolved class resource will not be checked for updates.
   * </p>
   * Returns <b>null</b> if this class loader is not registered with JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param className The public class name.
   * @param bytecode class bytes.
   * @param protectionDomain The ProtectionDomain of the class (may be <code>null</code>).
   * @param processedByLoader Whether given bytes should be processed by class loader in the same way
   *   as a normally loaded classes. Set to <code>false</code> when class is supposed to be define similarly to
   *   reflectively calling <code>ClassLoader.defineClass</code> that bypasses logic inside defining loader.
   * @return (Re)defined class or <code>null</code>.
   *
   * @since 7.0.1
   */
  public Class<?> defineReloadableClass(ClassLoader cl, String className, byte[] bytecode, ProtectionDomain protectionDomain,
      boolean processedByLoader);

  /**
   * Same as defineReloadableClass(cl, className, bytecode, protectionDomain, true)
   *
   * @param cl The custom class loader being integrated.
   * @param className The public class name.
   * @param bytecode class bytes.
   * @param protectionDomain The ProtectionDomain of the class (may be <code>null</code>).
   *
   * @return <code>true</code> if the class was managed by JRebel and redefined.
   * @since 3.0
   */
  public Class<?> defineReloadableClass(ClassLoader cl, String className, byte[] bytecode, ProtectionDomain protectionDomain);

  /**
   * Redefines a class managed by JRebel.
   * 
   * @param klass Class managed by JRebel.
   * @param bytecode class bytes.
   * @param processedByLoader Whether given bytes should be processed by class loader in the same way
   *   as a normally loaded classes. Set to <code>false</code> when class is supposed to be define similarly to
   *   reflectively calling <code>ClassLoader.defineClass</code> that bypasses logic inside defining loader.
   * @return <code>true</code> if the class was managed by JRebel and redefined.
   *
   * @since 7.0.1
   */
  public boolean redefineReloadableClass(Class<?> klass, byte[] bytecode, boolean processedByLoader);

  /**
   * Same as redefineReloadableClass(klass, bytecode, true)
   *
   * @param klass Class managed by JRebel.
   * @param bytecode class bytes.
   *
   * @return <code>true</code> if the class was managed by JRebel and redefined.
   *
   * @since 3.0
   */
  public boolean redefineReloadableClass(Class<?> klass, byte[] bytecode);

  /**
   * Redefines a hidden class managed by JRebel.
   * 
   * @param klass hidden class managed by JRebel.
   * @param bytecode class bytes.
   * @return the redefined hidden class.
   *
   * @since 2022.1.2
   */
  public Class<?> redefineReloadableHiddenClass(Class<?> klass, byte[] bytecode);

  /**
   * Redefine a set of classes managed by JRebel.
   * @param classDefinitions Array of ClassDefinition objects containing the Class and new class-bytes
   */
  public void redefineClasses(ClassDefinition... classDefinitions);
  
  /**
   * Returns <code>true</code> if the given resource is managed by JRebel. 
   * <p>
   * In that case resources can be resolved using the corresponding methods of the <code>Integration</code>.
   * If any of these methods returns a <code>null</code> it must be returned by the <code>ClassLoader</code> as well.
   * 
   * @param cl The custom class loader being integrated.
   * @param name The public resource name.
   * @return <code>true</code> if the given resource is managed by JRebel.
   */
  public boolean isResourceReplaced(ClassLoader cl, String name);
  
  /**
   * Resolves resources managed by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param name The public resource name.
   * @return Resolved resource URL or <b>null</b>.
   */
  public URL findResource(ClassLoader cl, String name);

  /**
   * Resolves resources only managed by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param name The public resource name.
   * @return Resolved resource URL or <b>null</b>.
   */
  public URL findRebelResource(ClassLoader cl, String name);

  /**
   * Resolves resources managed by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param name The public resource name.
   * @return Resolved resource URLs or <b>null</b>.
   */
  public Enumeration<URL> findResources(ClassLoader cl, String name);
  
  /**
   * Resolves resources only managed by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param name The public resource name.
   * @return Resolved resource URLs or <b>null</b>.
   */
  public Enumeration<URL> findRebelResources(ClassLoader cl, String name);

  /**
   * Returns the original resources with given name.
   * This is used to get the original result unaltered by JRebel.
   * 
   * @param cl The custom class loader being integrated.
   * @param name The resource name.
   * @return Resolved resource URLs or <b>null</b>.
   */
  public Enumeration<URL> getTransparentResources(ClassLoader cl, String name);

  /**
   * Returns the original resource with given name.
   * This is used to get the original result unaltered by JRebel.
   *
   * @param cl The custom class loader being integrated.
   * @param name The resource name.
   * @return Resolved resource URL or <b>null</b>.
   */
  public URL getTransparentResource(ClassLoader cl, String name);

  /**
   * Record location of found class. This method may be used to improve cache
   * hit ration for overhead resource requests in class loaders whose findClass
   * does not call findResource.
   * 
   * @param cl The custom class loader being integrated.
   * @param className The public class name.
   * @param url URL of class resource.
   * 
   * @since 4.6
   */
  public void recordClassResource(ClassLoader cl, String className, URL url);

  public void recordClassResource(ClassLoader cl, String className, URL url, byte[] bytes);

  /**
   * Cache results for class and resource request separately. Use when the result
   * of resource lookup as returned by findResource may be different from what
   * is recorded with recordClassResource.
   * 
   * @param cl The custom class loader being integrated.
   */
  public void useSeparateCacheForClassesAndResources(ClassLoader cl);

  /**
   * Adds a bytecode processor for the given class in the given classloader. It will be run before that class is loaded 
   * and is mainly use to enable integration with a specific framework or classloader. Allows to additionally
   * restrict the bytecode processing to the given classloader.
   *
   * @param cl the ClassLoader for the class
   * @param className the target class by name
   * @param processor the processor to add
   */
  public void addIntegrationProcessor(ClassLoader cl, String className, ClassBytecodeProcessor processor);
  
  /**
   * Adds a bytecode processor for the given class. It will be run before that class is loaded 
   * and is mainly use to enable integration with a specific framework or classloader.
   *
   * @param className the target class by name
   * @param processor the processor to add
   */
  public void addIntegrationProcessor(String className, ClassBytecodeProcessor processor);
  
  /**
   * Adds a bytecode processor for the given classloader. It will be run before that class is loaded 
   * and is mainly use to enable integration with a specific framework or classloader. Allows to additionally
   * restrict the bytecode processing to the given classloader.
   *
   * @param cl the ClassLoader for the classes
   * @param processor the processor to add
   */
  public void addIntegrationProcessor(ClassLoader cl, ClassBytecodeProcessor processor);
  
  /**
   * Adds a bytecode processor for all classes. It will be run before that class is loaded 
   * and is mainly used for integration with preprocessors like AspectJ.
   * <p>
   * If <code>managedOnly</code> is <code>true</code> the processor will only be run for the 
   * classes managed by JRebel. This is useful to enable processing usually done in the 
   * class loader <code>findClass()</code> method, e.g. the JBoss AOP.
   *
   * @param processor the processor to add
   * @param managedOnly should we only run it on reloadable classes
   */
  public void addIntegrationProcessor(ClassBytecodeProcessor processor, boolean managedOnly);

  /**
   * AAdds a bytecode processor for the given classloader. It will be run before that class is loaded 
   * and is mainly used for integration with preprocessors like AspectJ.
   * <p>
   * If <code>managedOnly</code> is <code>true</code> the processor will only be run for the 
   * classes managed by JRebel. This is useful to enable processing usually done in the 
   * class loader <code>findClass()</code> method, e.g. the JBoss AOP.
   *
   * @param cl the ClassLoader for the classes
   * @param processor the processor to add
   * @param managedOnly should we only run it on reloadable classes
   */
  public void addIntegrationProcessor(ClassLoader cl, ClassBytecodeProcessor processor, boolean managedOnly);

  /**
   * Adds a bytecode processor for the set of given classes. It will be run before those classes are loaded 
   * and is mainly use to enable integration with a specific framework or classloader.
   *
   * @param classNames the class to associate the processor with
   * @param processor the processor to add
   * 
   * @see #addIntegrationProcessor(String, ClassBytecodeProcessor)
   */
  public void addIntegrationProcessor(String[] classNames, ClassBytecodeProcessor processor);
  
  /**
   * Adds a bytecode processor for the set of given classes in the given classloader. It will be run before those classes are loaded 
   * and is mainly use to enable integration with a specific framework or classloader. Allows to additionally
   * restrict the bytecode processing to the given classloader.
   *
   * @param cl the ClassLoader for the classes
   * @param classNames the class to associate the processor with
   * @param processor the processor to add
   * 
   * @see #addIntegrationProcessor(ClassLoader, String, ClassBytecodeProcessor)
   */
  public void addIntegrationProcessor(ClassLoader cl, String[] classNames, ClassBytecodeProcessor processor);

  /**
   * Remove bytecode processor. 
   *
   * @param processor the processor to remove
   */
  public void removeIntegrationProcessor(ClassBytecodeProcessor processor);

  /**
   * Remove bytecode processors that are loaded by the ClassLoader
   *
   * @param cl the ClassLoader to remove all processors from
   */
  public void removeIntegrationProcessors(ClassLoader cl);

  /**
   * Adds a bytecode processor for the given class in boot classloader. It will be run before that class is loaded 
   * and is mainly use to enable integration with a specific framework or classloader.
   *
   * @param className the class to associate the processor with
   * @param processor the processor to add
   */
  public void addBootClassProcessor(String className, ClassBytecodeProcessor processor);

  /**
   * Adds a bytecode processor for set of given classes in boot classloader. It will be run before that class is loaded
   * and is mainly use to enable integration with a specific framework or classloader.
   *
   * @param classNames the classes to associate the processor with
   * @param processor the processor to add
   *
   * @see #addBootClassProcessor(String, ClassBytecodeProcessor)
   */
  public void addBootClassProcessor(String[] classNames, ClassBytecodeProcessor processor);

  /**
   * Adds a bytecode processor in boot classloader. It is mainly use to enable integration with a specific framework or classloader.
   * This API is only available when JRebel is configured as java agent.
   *
   * @param classBytecodeProcessor the processor to add
   */
  public void addBootClassProcessor(ClassBytecodeProcessor classBytecodeProcessor);

  /**
   * Report CBP failure, implementation must be async.
   * 
   * @param e the error to report
   * @param version version
   * @param jarFile location
   */
  void reportError(Throwable e, String version, String jarFile);

  /**
   * Registers a call back invoked when a given class loader is being destroyed.
   * 
   * @param cl The custom class loader being integrated.
   * @param listener class loader destruction listener.
   */
  public void addClassLoaderDestructionListener(ClassLoader cl, ClassLoaderDestructionListener listener);

  /**
   * Removes a call back invoked when a given class loader is being destroyed.
   * 
   * @param listener class loader destruction listener.
   */
  public void removeClassLoaderDestructionListener(ClassLoaderDestructionListener listener);

  /**
   * Returns <code>true</code> if
   * {@link Plugin#checkDependencies(ClassLoader, ClassResourceSource)}
   * is being called for the given class loader.
   * <p>
   * During the given process no class path entries should be altered.
   *
   * @param cl the target ClassLoader
   * @return <code>true</code> if {@link Plugin#checkDependencies(ClassLoader, ClassResourceSource)} is being called for the given class loader.
   */
  public boolean isCheckingPluginDependencies(ClassLoader cl);

  /**
   * Disable running plugins in given class loader.
   *
   * @param cl the target ClassLoader
   */
  public void disablePlugins(ClassLoader cl);

  /**
   * Disable class reloading in given class loader. Resource management will still be enabled.
   *
   * @param cl the target ClassLoader
   */
  public void disableReloading(ClassLoader cl);

  /**
   * Disable class reload check in given class loader. Resource management will still be enabled.
   *
   * @param cl the target ClassLoader
   */
  public void disableReloadDetection(ClassLoader cl);

  /**
   * Registers a callback that will be invoked after the main method is called.
   * Callback will be invoked in the same order as they were added.
   * If a callback is added after the main method has been called it will be invoked immediately.
   *
   * @param callback the callback
   */
  public void addAfterMainCallback(AfterMainCallback callback);

  /**
   * Disables the SecurityManager in the current thread.
   */
  public void disableSecurityManager();

  /**
   * Re-enables the disabled SecurityManager in the current thread.
   */
  public void enableSecurityManager();

  /**
   * Returns an instance of {@link ClassBytecodeProcessorCache} which can be used by class bytecode processors to cache
   * their transformations between runs.
   *
   * @return cacheCleaner the cache
   */
  ClassBytecodeProcessorCache getClassBytecodeProcessorCache();

  /**
   * Add reference from given class loader to given object. This will prevent GC from collecting the input object before
   * class loader is collected.
   *
   * @param cl Class loader that will be used to keep object alive.
   * @param value Object that will be made to live as long as class loader.
   *
   * @since 2019.2.2
   */
  void bindToClassLoader(ClassLoader cl, Object value);

  /**
   * An integration point to add once per reload caches that should be cleared after reload.
   *
   * @param cacheCleaner a listener to be called for cleanup
   *
   * @since 2020.1.2
   */
  public void addCacheCleaner(CacheCleaner cacheCleaner);
}