/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

/**
 * <p>Provides notification services for request/response based containers.
 * To use acquire an instance from the {@link RequestIntegrationFactory}.</p>
 * <p>
 * The life-cycle of executing requests on any servers should be the following:
 * <ol>
 *   <li>Parse the request and prepare the environment for the given web application as usually.</li>
 *   <li>Invoke {@link #fireRawRequest(RebelServletContext, Object, Object)}</li>
 *   <li>If <code>false</code> is received do the following:
 *     <ol>
 *       <li>Invoke {@link #fireBeforeRequest(RebelServletContext)}.</li>
 *       <li>Handle the request as usually by executing a Filter chain or a Servlet directly.</li>
 *       <li>Invoke {@link #fireRequestFinally(RebelServletContext)} (even if an error occurred).</li>
 *     </ol>
 *   </li>
 *   <li>Finish processing the request as usually.</li>
 * </ol>
 *
 * @author Igor Malinin
 * @author Rein Raudjärv
 *
 * @see RequestIntegrationFactory
 * @see RequestListener
 * @see Plugin
 */
public interface RequestIntegration {

  /* Request hooks to be implemented on all servers */

  /**
   * This must be called before the server handles an HTTP request
   * to give a chance to request listeners to fully manage this request.
   * <p>
   * The <code>request</code> and <code>response</code> objects must be valid
   * and ready to be used for rendering content, redirecting the request etc.
   * </p>
   * <p>
   * This method must be invoked before any <code>javax.servlet.Filter</code> or
   * <code>javax.servlet.Servlet</code> is invoked.
   * </p>
   * <p>
   * If <code>true</code> is returned the server itself must not manage
   * this request. In this case {@link #fireBeforeRequest(RebelServletContext)}
   * and {@link #fireRequestFinally(RebelServletContext)} must not be called also.
   *
   * @param context the servlet context of this web application.
   * @param request an instance of <code>javax.servlet.http.HttpServletRequest</code>.
   * @param response an instance of <code>javax.servlet.http.HttpServletResponse</code>.
   *
   * @return true if any request listener has managed this request.
   *
   * @since 3.6
   */
  boolean fireRawRequest(RebelServletContext context, Object request, Object response);

  /**
   * This must be called before the server handles an HTTP request.
   * <br>
   * <p>
   * It must be also invoked before any <code>javax.servlet.Filter</code> or
   * <code>javax.servlet.Servlet</code> is invoked.
   * However the environment of the given web application must be active.
   * E.g. <code>Thread.currentThread().getContextClassLoader()</code> must
   * return the application class loader.
   * In secure environments the corresponding <code>AccessControlContext</code>
   * must be set etc.
   * </p>
   * <br>
   * This must only be called if no request listener has managed this request.
   *
   * @param context the servlet context of this web application.
   *
   * @since 3.6
   */
  void fireBeforeRequest(RebelServletContext context);

  /**
   * This must be called at the end of the request (even if an exception occurred).
   * <p>
   * This must only be called if no request listener has managed this request.
   *
   * @param context the servlet context of this web application.
   *
   * @since 3.6
   */
  void fireRequestFinally(RebelServletContext context);


  /* Registering listeners */

  /**
   * Add a request listener for a web application.
   *
   * @param listener the listener to add
   * @param context the servlet context of this web application.
   *
   * @since 3.6
   */
  void addRequestListener(RebelServletContext context, RequestListener listener);

  /**
   * Remove a request listener of a web application.
   *
   * @param listener the listener to remove
   * @param context the servlet context of this web application.
   *
   * @since 3.6
   */
  void removeRequestListener(RebelServletContext context, RequestListener listener);

  /**
   * Add a global request listener.
   * <p>
   * Global request listeners are invoked before web application specific listeners.
   *
   * @param listener the listener to add
   *
   * @since 3.6
   */
  void addRequestListener(RequestListener listener);

  /**
   * Remove a global or a web application(s) specific request listener.
   *
   * @param listener the listener to remove
   * 
   * @since 3.6
   */
  void removeRequestListener(RequestListener listener);
}
