/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

/**
 * An HTTP request listener.
 *
 * @see RequestIntegration
 *
 * @author Rein Raudjärv
 *
 * @since 3.6
 */
public interface RequestListener {

  /**
   * Give a chance to this listener to fully manage this request.
   * <p>
   * If <code>true</code> is returned neither the server itself nor
   * any other request listener have a chance to manage this request.
   * </p>
   * <p>
   * Any exceptions will be caught and logged to the standard output.
   *
   * @param context an instance of <code>org.zeroturnaround.javarebel.RebelServletContext</code>.
   * @param request an instance of <code>javax.servlet.http.HttpServletRequest</code>.
   * @param response an instance of <code>javax.servlet.http.HttpServletResponse</code>.
   *
   * @throws Exception any Exception
   * @return true if this listener has managed this request.
   */
  default boolean rawRequest(Object context, Object request, Object response) throws Exception {
    return false;
  }

  /**
   * Called before the server handles an HTTP request.
   * <p>
   * Only called if no listener managed this request.
   * </p>
   * <p>
   * Any exceptions will be caught and logged to the standard output.
   *
   * @throws Exception any Exception
   */
  default void beforeRequest() throws Exception {
  }

  /**
   * Called at the end of the request (even if an exception occurred).
   * <p>
   * Only called if no listener managed this request.
   * </p>
   * <p>
   * Any exceptions will be caught and logged to the standard output.
   *
   * @throws Exception any Exception
   */
  default void requestFinally() throws Exception {
  }

  /**
   * Returns the priority of the listener that defines the order of listener invocation.
   * First run the listener with greatest priority value. Negative values are the latest.
   *
   * @return the priority of the listener
   */
  default int priority() {
    return 0;
  }
}