/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.support;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;

import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.Resource;

/**
 * A file system resource.
 * 
 * @author Jevgeni Kabanov (ekabanov@zeroturnaround.com)
 */
public class FileResource implements Resource {
  
  private final File file;
  private final URL url;

  public FileResource(File file) {
    this.file = file;
    try {
      this.url = file.toURI().toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }
  
  public FileResource(URL url) {
    this.url = url;
    this.file = ResourceUtils.getFile(url);
  }

  public File getFile() {
    return file;
  }

  public byte[] getBytes() {
    long length = file.length();
    if (length > Integer.MAX_VALUE) {
      LoggerFactory.getLogger("Core").error("File '" + file.getAbsolutePath() + "' is too large.");
      return null;
    }

    try {
      InputStream is = new FileInputStream(file);
      try {
        byte[] bytes = new byte[(int)length];

        int offset = 0;
        int numRead;
        while ((numRead = is.read(bytes, offset, bytes.length - offset)) != -1) {
          offset += numRead;
          if (offset == bytes.length) {
            break;
          }
        }

        if (offset < bytes.length || is.read() != -1) {
          LoggerFactory.getLogger("Core").error("File changed while reading " + file.getAbsolutePath());
          return null;
        }

        return bytes;
      }
      finally {
        is.close();
      }
    }
    catch (IOException e) {
      LoggerFactory.getLogger("Core").error(e);
      return null;
    }
  }

  public long lastModified() {
    return file.lastModified();
  }

  public String toString() {
    return url.toString();
  }

  public URL toURL() {
    return url;
  }
  
}
