/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.support;

import java.io.File;

import org.zeroturnaround.javarebel.FileEventListener;

/**
 * Wrapper for filtering out events for only the specified file.
 */
public class FileNameAwareListener implements FileEventListener {
  private final FileEventListener target;
  private final String fileName;

  /**
   * @param listener event listener
   * @param fileName name of monitored file or null. If null all events in
   *          monitored directory will be forwarded to target listener.
   */
  public FileNameAwareListener(FileEventListener listener, String fileName) {
    this.target = listener;
    this.fileName = fileName;
  }

  public FileEventListener getParent() {
    return target;
  }

  public boolean isRecursive() {
    return false;
  }

  public void onFileAdd(File file) {
    if (accept(file)) {
      target.onFileAdd(file);
    }
  }

  public void onFileRemove(File file) {
    if (accept(file)) {
      target.onFileRemove(file);
    }
  }

  public void onFileChange(File file) {
    if (accept(file)) {
      target.onFileChange(file);
    }
  }

  private boolean accept(File file) {
    return fileName == null || fileName.equals(file.getName());
  }

  public void onFileDirty(File file) {
    target.onFileDirty(file);
  }

  public void onFailure() {
    target.onFailure();
  }
}