package org.zeroturnaround.javarebel;

/**
 * This interface allows {@link ClassBytecodeProcessor class bytecode processors} to cache their transformations between
 * runs. A typical usage will look something like this:
 * <pre>
 *   public class MyCBP implements ClassBytecodeProcessor {
 *     public byte[] process(ClassLoader cl, String classname, byte[] bytecode) {
 *       ClassBytecodeProcessorCache cache = IntegrationFactory.getInstance().getClassBytecodeProcessorCache();
 *       String key = cache.computeKey(cl, classname, bytecode, null);
 *       if (key != null) {
 *         byte[] cachedBytes = cache.get(key);
 *         if (cachedBytes != null)
 *           return cachedBytes;
 *       }
 *
 *       // ... do actual processing ...
 *
 *       if (key != null)
 *         cache.put(key, processedBytes);
 *       return processedBytes;
 *     }
 *   }
 * </pre>
 */
public interface ClassBytecodeProcessorCache {

  /**
   * Computes the cache key for the given arguments.
   * @param className the name of the processed class
   * @param bytecode the class bytes that are to be processed
   * @param processor the processor
   * @param tag a custom tag to differentiate distinct processor configurations
   * @return a cache key, or {@code null} if the class loader is not cache capable
   */
  String computeKey(String className, byte[] bytecode, ClassBytecodeProcessor processor, String tag);

  /**
   * @param key the key
   * 
   * @return the cached bytes corresponding to the given key.
   */
  byte[] get(String key);

  /**
   * Stores given bytes associated with the given key.
   * 
   * @param key the key
   * @param bytes the bytes to cache
   */
  void put(String key, byte[] bytes);
}
