/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

/**
 * <p>This class manages JRebel configuration.</p>
 * 
 * <p>To manage the classes that will be instrumented use the filter methods 
 * {@link #addExcludeManagedFilter(ClassFilter)} and {@link #addIncludeManagedFilter(ClassFilter)}. 
 * If you need to override the default behavior (e.g. you don't need class files to be managed)
 * use {@link #clearManagedFilters()} to disable the default filters. If you want to filter 
 * classes by packages you can use {@link org.zeroturnaround.javarebel.support.PackageClassFilter}.</p>
 * 
 * <p>The filter methods work as follows:</p>
 * 
 * <ul>
 *   <li>If at least one exclude filter matches a class it will not be instrumented.</li>
 *   <li>If none of the exclude filters match a class and at least one of the include filters matches a class it will be instrumented.</li>
 * </ul>
 * 
 * <p>For example, if you want all classes in a particular JAR to be instrumented you 
 * can use a following filter:</p>
 * 
 * <pre>
 *   ConfigurationFactory.getInstance()
 *     .addIncludeManagedFilter(new ClassFilter() {
 *     public boolean matches(ClassLoader cl, String className,
 *         Resource classResource) {        
 *       return classResource.toURL().getPath().contains("mylibrary.jar");
 *     }
 *   });
 * </pre>
 * 
 * @since 1.2
 * 
 * @author Jevgeni Kabanov (ekabanov@zeroturnaround.com)
 * 
 * @see ConfigurationFactory
 * @see Integration
 * @see Reloader
 * @see Logger
 * @see Plugin
 */
public interface Configuration {
  
  /**
   * Gets the system property indicated by the specified key.
   * 
   * If no system property is found with that key
   * a property from file named <code>jrebel.properties</code>
   * is returned.
   * 
   * @param      key   the name of the system property.
   * @return     the string value of the property,
   *             or <code>null</code> if there is no property with that key.
   */
  String getProperty(String key);
  
  /**
   * Gets the boolean value of a system property indicated by the specified key.
   * 
   * If no system property is found with that key
   * a property from file named <code>jrebel.properties</code>
   * is returned.
   * 
   * @param      key   the name of the system property.
   * @return     the boolean value of the property,
   *             <code>false</code> if there is no property with that key.
   */
  boolean getBoolean(String key);

  /**
   * Gets the boolean value of a system property indicated by the specified key.
   *
   * If no system property is found with that key
   * a property from file named <code>jrebel.properties</code>
   * is returned.
   *
   * @param      key   the name of the system property.
   * @param      defaultValue value to return if there is no property with that key
   * @return     the boolean value of the property or <code>defaultValue</code> .
   */
  boolean getBoolean(String key, boolean defaultValue);

  /**
   * Advisable time between checks for updates in framework specific configuration
   * files. 
   * 
   * @return time in ms.
   */
  int getCheckInterval();

  /**
   * @return product name (JRebel).
   */
  String getProductName();
  
  /**
   * Checks whether the given plugin is enabled.
   * 
   * @param id plugin id (see {@link Plugin#getId()}).
   * @return <code>true</code> if the given plugin is enabled.
   */
  boolean isPluginEnabled(String id);
  
  /**
   * Clears all currently used filters including the default ones. 
   * Use it if you need to override the default behavior (e.g. you don't need class files to be managed).
   */
  void clearManagedFilters();
  
  /**
   * Adds an exclude filter for the entire class loader (including plug-ins).
   */
  void addExcludeFilter(ClassLoader cl);
  
  /**
   * Adds an exclude filter for the entire class loader (including plug-ins).
   */
  void addExcludeFilter(String classLoaderClassName);
  
  /**
   * Adds an exclude filter for entire class loaders (including plug-ins).
   */
  void addExcludeFilter(ClassLoaderFilter classLoaderFilter);
  
  /**
   * Returns whether the filters will deny the given class loader.
   */
  boolean isClassLoaderExcluded(ClassLoader cl);  
  
  /**
   * Adds an include filter for managed classes. 
   */
  void addIncludeManagedFilter(ClassLoader cl);
  
  /**
   * Adds an exclude filter for managed classes. 
   */
  void addExcludeManagedFilter(ClassLoader cl);
  
  /**
   * Adds an include filter for managed classes. 
   */
  void addIncludeManagedFilter(ClassIdentityFilter classFilter);

  /**
   * Adds an exclude filter for managed classes. 
   */
  void addExcludeManagedFilter(ClassIdentityFilter classFilter);
  
  /**
   * Adds an include filter for managed classes. 
   * If none of the exclude filters match a class and at least one 
   * of the include filters matches a class it will be instrumented by JRebel.
   */
  void addIncludeManagedFilter(ClassFilter classFilter);
  
  /**
   * Adds an exclude filter for managed classes. 
   * If at least one exclude filter matches a class it will not be instrumented by JRebel.
   */
  void addExcludeManagedFilter(ClassFilter classFilter);
  
  /**
   * Returns whether the filters will allow the given class to be managed.
   * This involves both <code>ClassFilter</code> and <code>ClassIdentityFilter</code> rules.
   */
  boolean isManagedClass(ClassLoader cl, String classname, Resource cr);

  /**
   * Returns whether the filters will deny the given class to be managed.
   * This involves only <code>ClassIdentityFilter</code> rules.
   */
  boolean isManagedClassExcluded(ClassLoader cl, String classname);

  /**
   * Returns a JRebel Resource exactly as JRebel will load it. 
   * E.g. if the resource is in the monitored directory it will return that. 
   * Note that this already includes managed filtering, that is if
   * {@link #isManagedClass(ClassLoader, String, Resource)} returns false
   * then this method will return null.
   */
  Resource getClassResource(ClassLoader cl, String classname);

}
