/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.generic;

import java.lang.ref.WeakReference;
import java.net.URL;

import org.zeroturnaround.javarebel.IntegrationFactory;
import org.zeroturnaround.javarebel.Resource;
import org.zeroturnaround.javarebel.integration.support.BaseClassResourceSource;
import org.zeroturnaround.javarebel.integration.util.ResourceUtil;

/**
 * Default class-to-resource resolution strategy.
 * 
 * <p>The <code>findResource</code> and <code>findResources</code> methods
 * of the given class loader are used.
 * 
 * @see ClassLoader
 * @see org.zeroturnaround.javarebel.integration.util.ResourceUtil#findResource(ClassLoader, String)
 * @see org.zeroturnaround.javarebel.integration.util.ResourceUtil#findResources(ClassLoader, String)
 */
public class FindResourceClassResourceSource extends BaseClassResourceSource {
  
  private WeakReference<ClassLoader> classloader;
  
  public FindResourceClassResourceSource(ClassLoader classloader) {
    this.classloader = new WeakReference<ClassLoader>(classloader);
  }

  private ClassLoader getClassloader() {
    return classloader.get();
  }

  public Resource getLocalResource(String name) {
    try {
      URL url = ResourceUtil.findResource(getClassloader(), name);
      return ResourceUtil.asResource(url);
    }
    catch (RuntimeException e) {
      // This class is used to find resources from rebel ClassLoaders
      // When an app undeployes we sometimes unregister the ClassLoader and no reloading is done in there.
      // Now to avoid log exceptions from such closed ClassLoaders we just return no resources instead of exceptions from them.
      if (!IntegrationFactory.getInstance().isRegisteredClassLoader(getClassloader())) {
        return null;
      }
      throw e;
    }
  }

  public Resource[] getLocalResources(String name) {
    try {
      URL[] urls = ResourceUtil.findResources(getClassloader(), name);
      return ResourceUtil.asResources(urls);
    }
    catch (RuntimeException e) {
      // This class is used to find resources from rebel ClassLoaders
      // When an app undeployes we sometimes unregister the ClassLoader and no reloading is done in there.
      // Now to avoid log exceptions from such closed ClassLoaders we just return no resources instead of exceptions from them.
      if (!IntegrationFactory.getInstance().isRegisteredClassLoader(getClassloader())) {
        return null;
      }
      throw e;
    }
  }
}