/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.generic;

import java.lang.ref.WeakReference;
import java.net.URL;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import org.zeroturnaround.javarebel.Resource;
import org.zeroturnaround.javarebel.integration.support.BaseClassResourceSource;
import org.zeroturnaround.javarebel.integration.util.ResourceUtil;

/**
 * Class resource source using privileged actions. 
 * 
 * @author Rein Raudjärv
 * 
 * @see RestrictedResource
 */
public class RestrictedClassResourceSource extends BaseClassResourceSource {

  private final AccessControlContext acc;
  private final WeakReference<ClassLoader> classloader;
  
  public RestrictedClassResourceSource(ClassLoader classloader) {
    this.classloader = new WeakReference<ClassLoader>(classloader);
    this.acc = AccessController.getContext();
  }
  
  public Resource getLocalResource(final String name) {
    URL url = AccessController.doPrivileged(new PrivilegedAction<URL>() {
      public URL run() {
        return ResourceUtil.findResource(classloader.get(), name);
      }
    }, acc);
    
    if (url == null)
      return null;
    
    return RestrictedResource.newInstance(acc, ResourceUtil.asResource(url));
  }

  public Resource[] getLocalResources(final String name) {
    URL[] urls = AccessController.doPrivileged(new PrivilegedAction<URL[]>() {
      public URL[] run() {
        return ResourceUtil.findResources(classloader.get(), name);
      }
    }, acc);
    
    if (urls == null)
      return null;
    
    Resource[] result = new Resource[urls.length];
    for (int i = 0; i < urls.length; i++) {
      URL url = urls[i];
      result[i] = RestrictedResource.newInstance(acc, ResourceUtil.asResource(url));
    }
    return result;
  }

}