/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.monitor;

import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.Resource;
import org.zeroturnaround.javarebel.StopWatch;
import org.zeroturnaround.javarebel.integration.util.MiscUtil;
import org.zeroturnaround.javarebel.integration.util.SecurityController;
import org.zeroturnaround.javarebel.integration.util.SecurityController.PrivilegedAction;
import org.zeroturnaround.javarebel.support.FileResource;

/**
 * @author Jevgeni Kabanov
 */
public class MonitoredResource {

  private static final Logger log = LoggerFactory.getLogger("MonitoredResource");

  public final Resource res;
  private long lastModified;
  private int hash = 0;

  public MonitoredResource(Resource res) {
    this.res = res;
    lastModified = res.lastModified();
    hash = getHash();
  }

  public synchronized boolean modified() {
    return SecurityController.doWithoutSecurityManager(new PrivilegedAction<Boolean>() {
      public Boolean run() {
        return isModified();
      }
    });
  }

  private boolean isModified() {
    long newLastModified = res.lastModified();
    if (newLastModified <= lastModified) {
      return false;
    }
    lastModified = newLastModified;
    if (hash == 0) {
      return true;
    }
    int newHash = getHash();
    if (hash == newHash) {
      return false;
    }
    hash = newHash;
    return true;
  }

  private int getHash() {
    if (lastModified == 0 || (res instanceof FileResource && ((FileResource) res).getFile().isDirectory())) {
      return 0;
    }
    StopWatch sw = log.createStopWatch("MonitoredResource#hash");
    try {
      byte[] bytes = res.getBytes();
      if (bytes == null) {
        return 0;
      }
      return MiscUtil.hash32(bytes);
    }
    finally {
      sw.stop();
    }
  }

  public String toString() {
    return res.toString();
  }
}