package org.zeroturnaround.javarebel.integration.support;

import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.NotFoundException;
import org.zeroturnaround.bundled.javassist.bytecode.Descriptor;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;

/**
 * The class pool used by JavassistClassBytecodeProcessor
 */
class RebelClassPool extends ClassPool {

  private static final Logger log = LoggerFactory.getLogger("SDK-CBP");

  private final JavassistClassBytecodeProcessor processor;
  private final String classname;
  private final boolean processingClassInDefaultPkg;

  public RebelClassPool(ClassPool systemClassPool, JavassistClassBytecodeProcessor processor, String classname) {
    super(systemClassPool);
    this.childFirstLookup = true;
    this.processor = processor;
    this.classname = classname;
    processingClassInDefaultPkg = isClassInDefaultPackage(classname);
  }

  public static void init() {
  }

  public CtClass get(String name) throws NotFoundException {
    String checked = name;
    if (checked.charAt(0) == '[') {
      checked = Descriptor.toClassName(checked);
    }
    while (checked.endsWith("[]")) {
      checked = checked.substring(0, checked.lastIndexOf('['));
    }
    if (classname.equals(checked) || processor.acceptPathAsPrimitive(checked)) {
      return super.get(name);
    }
    if (!processor.acceptPathAsClass(checked)) {
      throw new NotFoundException(name);
    }
    if (isClassInDefaultPackage(checked)) {
      if (processor.getImplicitClassNames() != null) {
        CtClass result = tryLoadFromImplicitImports(name, checked);
        if (result != null) {
          return result;
        }
      }
      if (!processingClassInDefaultPkg)
        throw new NotFoundException(name);
    }
    return super.get(name);
  }

  private CtClass tryLoadFromImplicitImports(String name, String checked) throws NotFoundException {
    String fullClassName = processor.getImplicitClassNames().get(name);
    if (fullClassName != null) {
      return super.get(fullClassName);
    }
    log.trace("Did not find package for {} from {}", checked, name);
    return null;
  }

  public void clearCache() {
    classes.clear();
  }

  /**
   * @return true if the class has no package definition
   */
  private boolean isClassInDefaultPackage(String classname) {
    return classname.lastIndexOf('.') == -1;
  }
}