/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.zeroturnaround.javarebel.ClassBytecodeProcessor;
import org.zeroturnaround.javarebel.ClassEventListener;
import org.zeroturnaround.javarebel.ClassLoaderDestructionListener;
import org.zeroturnaround.javarebel.IntegrationFactory;
import org.zeroturnaround.javarebel.ReloaderFactory;
import org.zeroturnaround.javarebel.integration.support.ClassBytecodeProcessorWrapper;

/**
 * Utility class for creating class loader local bindings.
 */
public class ClassLoaderLocalUtil {
  // Map ClassLoader -> WeakReference<LoaderAttachment>
  private static final Map<ClassLoader, WeakReference<LoaderAttachment>> attachments = Collections.synchronizedMap(new WeakIdentityHashMap<ClassLoader, WeakReference<LoaderAttachment>>(1));
  // in case attaching to class loader fails keep a strong reference here
  private static final Set<LoaderAttachment> failedAttachements = Collections.synchronizedSet(new HashSet<LoaderAttachment>(1));

  /**
   * Alias for bind(cbp, cbp.getClass().getClassLoader())
   */
  public static ClassBytecodeProcessor bind(ClassBytecodeProcessor cbp) {
    return bind(cbp, cbp.getClass().getClassLoader());
  }

  /**
   * Creates a strong reference from class loader to cbp that will be cleared when class loader
   * is destroyed.
   * 
   * @param cbp
   * @param cl
   * @return wrapped cbp that is strongly reachable from given class loader
   */
  public static ClassBytecodeProcessor bind(ClassBytecodeProcessor cbp, ClassLoader cl) {
    LoaderAttachment la = getAttachment(cl);
    ClassBytecodeProcessor result = new DestructibleClassBytecodeProcessorAdapter(cbp, la);
    la.addReference(result);
    return result;
  }

  /**
   * Alias for bind(cel, cel.getClass().getClassLoader())
   */
  public static ClassEventListener bind(ClassEventListener cel) {
    return bind(cel, cel.getClass().getClassLoader());
  }

  /**
   * Creates a strong reference from class loader to cel that will be cleared when class loader
   * is destroyed.
   * NB!
   * In your own code you should only use weak references to the returned ClassEventListener
   * For example you can use WeakUtil.weakCEL or ClassEventListenerUtil.bindContextClassLoader
   * Or you will be in a danger of leaking some ClassLoaders through the cel
   * 
   * @param cel
   * @param cl
   * @return wrapped cel that is strongly reachable from given class loader
   */
  public static ClassEventListener bind(ClassEventListener cel, ClassLoader cl) {
    LoaderAttachment la = getAttachment(cl);
    ClassEventListener result = new DestructibleClassEventListenerAdapter(cel, la);
    la.addReference(result);
    return result;
  }

  /**
   * @param cl
   * @return data holder for given class loader
   */
  private static LoaderAttachment getAttachment(ClassLoader cl) {
    synchronized (attachments) {
      WeakReference<LoaderAttachment> ref = attachments.get(cl);
      LoaderAttachment la = ref == null ? null : ref.get();
      if (la == null) {
        la = new LoaderAttachment();
        IntegrationFactory.getInstance().bindToClassLoader(cl, la);
        IntegrationFactory.getInstance().addClassLoaderDestructionListener(cl, WeakUtil.weak(la));
        attachments.put(cl, new WeakReference<LoaderAttachment>(la));
      }
      return la;
    }
  }

  private static class LoaderAttachment implements ClassLoaderDestructionListener {
    final List<Object> items = Collections.synchronizedList(new ArrayList<Object>(1));
    volatile boolean destroyed = false;
    
    public void addReference(Object o) {
      items.add(o);
    }

    public void onDestroy(ClassLoader cl) {
      destroyed = true;
      items.clear();
      failedAttachements.remove(this);
    }
  }

  private static class DestructibleClassBytecodeProcessorAdapter implements ClassBytecodeProcessor, ClassBytecodeProcessorWrapper {
    private final ClassBytecodeProcessor cbp;
    private final LoaderAttachment la;
    private final String identity;

    public DestructibleClassBytecodeProcessorAdapter(ClassBytecodeProcessor cbp, LoaderAttachment la) {
      this.cbp = cbp;
      this.la = la;
      this.identity = MiscUtil.identityToString(this) + "[" + MiscUtil.dumpToString(cbp) + "]";
    }

    public byte[] process(ClassLoader cl, String classname, byte[] bytecode) {
      if (la.destroyed) {
        IntegrationFactory.getInstance().removeIntegrationProcessor(this);
        return bytecode;
      }

      return cbp.process(cl, classname, bytecode);
    }

    public int priority() {
      return cbp.priority();
    }

    public String toString() {
      return identity;
    }

    public ClassBytecodeProcessor getDelegate() {
      return cbp;
    }
  }

  private static class DestructibleClassEventListenerAdapter implements ClassEventListener {
    private final ClassEventListener cel;
    private final LoaderAttachment la;
    private final int priority;
    private final String identity;

    public DestructibleClassEventListenerAdapter(ClassEventListener cel, LoaderAttachment la) {
      this.cel = cel;
      this.la = la;
      this.priority = cel.priority();
      this.identity = MiscUtil.identityToString(this) + "[" + MiscUtil.dumpToString(cel) + "]";
    }

    public void onClassEvent(int eventType, Class<?> klass, Collection<ClassEventListener.ChangeType> changeTypes) throws Exception {
      if (la.destroyed) {
        ReloaderFactory.getInstance().removeClassReloadListener(this);
        return;
      }

      cel.onClassEvent(eventType, klass, changeTypes);
    }

    public int priority() {
      return priority;
    }

    public String toString() {
      return identity;
    }
  }

}
