/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.io.File;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;

import org.zeroturnaround.javarebel.ClassBytecodeProcessor;
import org.zeroturnaround.javarebel.ClassEventListener;
import org.zeroturnaround.javarebel.ClassIdentityFilter;
import org.zeroturnaround.javarebel.ClassLoaderDestructionListener;
import org.zeroturnaround.javarebel.FileEventListener;
import org.zeroturnaround.javarebel.IntegrationFactory;
import org.zeroturnaround.javarebel.ReloaderFactory;
import org.zeroturnaround.javarebel.RequestIntegrationFactory;
import org.zeroturnaround.javarebel.RequestListener;
import org.zeroturnaround.javarebel.ResourceIntegrationFactory;
import org.zeroturnaround.javarebel.integration.support.ClassBytecodeProcessorWrapper;

/**
 * @author Rein Raudjärv
 */
public class WeakUtil {

  /**
   * @param elements the elements.
   * @return <code>List&lt;Reference&gt;</code> the given elements
   *   wrapped into <code>WeakReferences</code>-s.
   */
  public static <T> List<WeakReference<T>> createWeakReferenceList(Collection<T> elements) {
    if (elements == null)
      return null;

    List<WeakReference<T>> result = new ArrayList<WeakReference<T>>();
    for (T object : elements) {
      if (object != null) {
        result.add(new WeakReference<T>(object));
      }
    }
    return result;
  }

  /**
   * @return the given elements in a weak set.
   */
  public static <T> Set<T> createWeakHashSet() {
    return createWeakHashSetInternal(null);
  }

  /**
   * @return the given elements in a weak identity set.
   */
  public static <T> Set<T> createWeakIdentityHashSet() {
    return createWeakHashSetInternal(new WeakIdentityHashMap<T, Boolean>(), null);
  }

  /**
   * @param elements the elements.
   * @return the given elements in a weak set.
   */
  public static <T> Set<T> createWeakHashSet(Collection<T> elements) {
    if (elements == null)
      return null;
    return createWeakHashSetInternal(elements);
  }

  private static <T> Set<T> createWeakHashSetInternal(Collection<T> elements) {
    return createWeakHashSetInternal(new WeakHashMap<T, Boolean>(), elements);
  }

  private static <T> Set<T> createWeakHashSetInternal(final Map<T, Boolean> map, Collection<T> elements) {
    Set<T> result = Collections.newSetFromMap(map);
    if (elements != null)
      result.addAll(elements);
    return result;
  }

  /**
   * Alias for weak(ClassBytecodeProcessor)
   */
  public static ClassBytecodeProcessor weakCBP(ClassBytecodeProcessor cbp) {
    return weak(cbp);
  }

  /**
   * @param cbp
   * @return wrapper from given cbp that references the original cbp weakly
   */
  public static ClassBytecodeProcessor weak(ClassBytecodeProcessor cbp) {
    return new WeakClassBytecodeProcessorAdapter(cbp);
  }

  /**
   * Alias for weak(ClassEventListener)
   */
  public static ClassEventListener weakCEL(ClassEventListener cel) {
    return weak(cel);
  }

  /**
   * @param cel
   * @return wrapper from given cbp that references the original cel weakly
   */
  public static ClassEventListener weak(ClassEventListener cel) {
    return new WeakClassEventListenerAdapter(cel);
  }

  /**
   * Alias for weak(ClassLoaderDestructionListener)
   */
  public static ClassLoaderDestructionListener weakCDL(ClassLoaderDestructionListener cdl) {
    return weak(cdl);
  }

  /**
   * @param cdl
   * @return wrapper from given cdl that references the original cel weakly
   */
  public static ClassLoaderDestructionListener weak(ClassLoaderDestructionListener cdl) {
    return new WeakClassLoaderDestructionListener(cdl);
  }

  /**
   * Alias for weak(ClassIdentityFilter)
   */
  public static ClassIdentityFilter weakCIF(ClassIdentityFilter cif) {
    return weak(cif);
  }

  /**
   * @param cif
   * @return wrapper from given cdl that references the original cif weakly
   */
  public static ClassIdentityFilter weak(ClassIdentityFilter cif) {
    return new WeakClassIdentityFilterAdapter(cif);
  }

  /**
   * @param listener
   * @return wrapper from given listener that references the original listener weakly
   */
  public static RequestListener weak(RequestListener listener) {
    return new WeakRequestListener(listener);
  }

  /**
   * Alias for weak(RequestListener)
   */
  public static RequestListener weakRL(RequestListener listener) {
    return weak(listener);
  }

  /**
   * @param listener
   * @return wrapper from given listener that references the original listener weakly
   */
  public static FileEventListener weak(FileEventListener listener) {
    return new WeakFileEventListener(listener);
  }

  private static class WeakClassIdentityFilterAdapter implements ClassIdentityFilter {
    private final WeakReference<ClassIdentityFilter> cbp;
    private final String identity;

    public WeakClassIdentityFilterAdapter(ClassIdentityFilter cbp) {
      this.cbp = new WeakReference<ClassIdentityFilter>(cbp);
      this.identity = MiscUtil.identityToString(cbp);
    }

    public boolean matches(ClassLoader cl, String className) {
      ClassIdentityFilter delegate = cbp.get();
      if (delegate == null)
        return false;
      return delegate.matches(cl, className);
    }

    public String toString() {
      return identity;
    }
  }

  private static class WeakClassBytecodeProcessorAdapter implements ClassBytecodeProcessor, RemovableListener, ClassBytecodeProcessorWrapper {
    private final WeakReference<ClassBytecodeProcessor> cbp;
    private final int priority;
    private final String identity;

    public WeakClassBytecodeProcessorAdapter(ClassBytecodeProcessor cbp) {
      this.cbp = new WeakListenerReference<ClassBytecodeProcessor>(this, cbp);
      this.identity = MiscUtil.identityToString(this) + "[" + MiscUtil.dumpToString(cbp) + "]";
      this.priority = getPriority(cbp);
    }

    private int getPriority(ClassBytecodeProcessor cbp) {
      try {
        return cbp.priority();
      }
      catch (AbstractMethodError e) {
        return 0;
      }
    }

    public byte[] process(ClassLoader cl, String classname, byte[] bytecode) {
      ClassBytecodeProcessor delegate = cbp.get();
      if (delegate == null) {
        remove();
        return bytecode;
      }
      return delegate.process(cl, classname, bytecode);
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      IntegrationFactory.getInstance().removeIntegrationProcessor(this);
    }

    public ClassBytecodeProcessor getDelegate() {
      return cbp.get();
    }

    public int priority() {
      return priority;
    }
  }

  private static class WeakClassEventListenerAdapter implements ClassEventListener, RemovableListener {
    private final WeakReference<ClassEventListener> cel;
    private final int priority;
    private final String identity;

    public WeakClassEventListenerAdapter(ClassEventListener cel) {
      this.cel = new WeakListenerReference<ClassEventListener>(this, cel);
      this.priority = cel.priority();
      this.identity = "weak[" + MiscUtil.dumpToString(cel) + "]";
    }

    public void onClassEvent(int eventType, Class<?> klass, Collection<ClassEventListener.ChangeType> changeTypes) throws Exception {
      ClassEventListener delegate = cel.get();
      if (delegate == null) {
        remove();
        return;
      }
      delegate.onClassEvent(eventType, klass, changeTypes);
    }

    public int priority() {
      return priority;
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      ReloaderFactory.getInstance().removeClassReloadListener(this);
    }
  }

  private static class WeakClassLoaderDestructionListener implements ClassLoaderDestructionListener, RemovableListener {
    private final WeakReference<ClassLoaderDestructionListener> cdl;
    private final String identity;

    public WeakClassLoaderDestructionListener(ClassLoaderDestructionListener cdl) {
      this.cdl = new WeakListenerReference<ClassLoaderDestructionListener>(this, cdl);
      this.identity = MiscUtil.identityToString(cdl);
    }

    public void onDestroy(ClassLoader cl) {
      ClassLoaderDestructionListener delegate = cdl.get();
      if (delegate == null) {
        remove();
        return;
      }
      delegate.onDestroy(cl);
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      IntegrationFactory.getInstance().removeClassLoaderDestructionListener(this);
    }
  }

  private static class WeakRequestListener implements RequestListener, RemovableListener {
    private final WeakReference<RequestListener> ref;
    private final String identity;
    private final int priority;

    public WeakRequestListener(RequestListener target) {
      this.ref = new WeakListenerReference<RequestListener>(this, target);
      this.identity = "weak[" + MiscUtil.dumpToString(target) + "]";
      this.priority = target.priority();
    }

    public boolean rawRequest(Object context, Object request, Object response) throws Exception {
      RequestListener target = ref.get();
      if (target != null)
        return target.rawRequest(context, request, response);
      remove();
      return false;
    }

    public void beforeRequest() throws Exception {
      RequestListener target = ref.get();
      if (target != null)
        target.beforeRequest();
    }

    public void requestFinally() throws Exception {
      RequestListener target = ref.get();
      if (target != null)
        target.requestFinally();
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      RequestIntegrationFactory.getInstance().removeRequestListener(this);
    }

    public int priority() {
      return priority;
    }
  }

  private static class WeakFileEventListener implements FileEventListener, RemovableListener {
    private final WeakReference<FileEventListener> ref;
    private final String identity;

    public WeakFileEventListener(FileEventListener target) {
      ref = new WeakListenerReference<FileEventListener>(this, target);
      this.identity = MiscUtil.identityToString(this) + "[" + MiscUtil.dumpToString(target) + "]";
    }

    private FileEventListener get() {
      FileEventListener target = ref.get();
      if (target == null)
        remove();
      return target;
    }

    public boolean isRecursive() {
      FileEventListener target = get();
      if (target != null)
        return target.isRecursive();
      return false;
    }

    public void onFileAdd(File file) {
      FileEventListener target = get();
      if (target != null)
        target.onFileAdd(file);
    }

    public void onFileChange(File file) {
      FileEventListener target = get();
      if (target != null)
        target.onFileChange(file);
    }

    public void onFileRemove(File file) {
      FileEventListener target = get();
      if (target != null)
        target.onFileRemove(file);
    }

    public void onFileDirty(File file) {
      FileEventListener target = get();
      if (target != null)
        target.onFileDirty(file);
    }

    public void onFailure() {
      FileEventListener target = get();
      if (target != null)
        target.onFailure();
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      ResourceIntegrationFactory.getInstance().removeFileListener(this);
    }
  }

  public static interface RemovableListener {
    void remove();
  }

  public static class WeakListenerReference<T> extends WeakReference<T> {
    private static final ReferenceQueue<Object> rq = new ReferenceQueue<Object>();
    private final RemovableListener owner;

    static {
      final Runnable r = new Runnable() {
        public void run() {
          try {
            while (true) {
              WeakListenerReference<?> ref = (WeakListenerReference<?>) rq.remove();
              ref.owner.remove();
            }
          } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
          }
        }
      };
      SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<Void>() {
        public Void run() {
          Thread t = new Thread(r, "rebel-weak-reaper");
          t.setDaemon(true);
          t.setContextClassLoader(null);
          t.start();
          return null;
        }
      });
    }

    public WeakListenerReference(RemovableListener owner, T target) {
      super(target, rq);
      this.owner = owner;
    }
  }
}