/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.monitor;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;

import org.zeroturnaround.javarebel.ConfigurationFactory;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.RebelSource;
import org.zeroturnaround.javarebel.StopWatch;
import org.zeroturnaround.javarebel.integration.util.ResourceUtil;
import org.zeroturnaround.javarebel.integration.util.WeakIdentityHashMap;
import org.zeroturnaround.javarebel.support.ResourceUtils;

/**
 * This is a helper class that makes it easier to record and reproduce a complicated
 * configuration building process. It keeps track of resources that should trigger the
 * rebuilding of the configuration when updated.
 * 
 * The usage pattern follows those steps:
 * <ol>
 *   <li>
 *     When the configuration building starts, call <code>beginConf(builder)</code>, providing
 *     the instance responsible for building the configuration as an argument.
 *   </li>
 *   <li>
 *     During the building process, call <code>registerConf</code> for all the underlying
 *     configuration resources that you want to be monitored. (For example, if you are building
 *     a Hibernate's <code>SessionFactory</code> with <code>AnnotationConfiguration</code>, you'll
 *     be calling <code>registerConf(klass)</code> for all classes containing JPA annotations).
 *   </li>
 *   <li>
 *     When the configuration building is done, call <code>endConf</code>.
 *   </li>
 * </ol>
 * 
 * 
 * All classes that build such configurations have to implement the Builder interface
 * to be usable together with the MonitoredResourceManager. In normal circumstances,
 * this means we have look up these classes and bytecode-patch them to implement that
 * interface. The main user is currently the Hibernate plugin. See <code>LocalSessionFactoryBeanCBP</code>
 * in the Hibernate plugin for an example usage. 
 * 
 * @author Jevgeni Kabanov
 * @author Rein Raudjärv
 */
public class MonitoredResourceManager {

  private static final Logger log = LoggerFactory.getLogger("MonitoredResourceManager");

  private static final ThreadLocal<Object> builderLocal = new ThreadLocal<Object>();
  
  /**
   * When starting to build a configuration, the object that is doing the building has to
   * be set here.
   */
  public static void beginConf(Object builder) {
    builderLocal.set(builder);
  }
  
  /**
   * When the building of the configuration is done, be sure to call <code>endConf()</code>
   * for safety.
   */
  public static void endConf() {
    builderLocal.set(null);
  }
  public static Object getConf() {
    return builderLocal.get();
  }

  private static final Map<Object, MonitoredResourceManager> managersPerBuilder = Collections.synchronizedMap(new WeakIdentityHashMap<Object, MonitoredResourceManager>());
  private static final Map<Class<?>, URL> classToURL = Collections.synchronizedMap(new WeakHashMap<Class<?>, URL>());

  private static synchronized MonitoredResourceManager get(Object builder) {
    return managersPerBuilder.get(builder);
  }
  
  private static synchronized MonitoredResourceManager getOrCreate(Object builder) {
    MonitoredResourceManager manager = managersPerBuilder.get(builder);
    if (manager == null) {
      manager = new MonitoredResourceManager();
      managersPerBuilder.put(builder, manager);
    }
    return manager;
  }
  
  
  private static final long MODIFIED_CHECK_MIN_INTERVAL = ConfigurationFactory.getInstance().getCheckInterval();

  private final Map<URL, MonitoredResource> monitoredConfs = Collections.synchronizedMap(new HashMap<URL, MonitoredResource>());
  private final Set<File> monitoredDirs = new HashSet<File>();
  private final List<RebelSource> rebelSources = new ArrayList<RebelSource>();
  private long lastCheck = System.currentTimeMillis();

  /**
   * Find the MonitoredResourceManager instance that corresponds to that builder. Check for
   * modifications to resources registered to that instance.
   * 
   * @return a set of modified resources
   */
  public static Set<String> modified(Object builder) {
    MonitoredResourceManager manager = get(builder);
    return manager == null ? Collections.<String>emptySet() : manager.modified();
  }
  
  /**
   * Check for modifications to resources registered to that MonitoredResourceManager.
   * 
   * @return a set of modified resources
   */
  private synchronized Set<String> modified() {
    if (System.currentTimeMillis() < lastCheck + MODIFIED_CHECK_MIN_INTERVAL)
      return Collections.emptySet();

    Set<String> result = new HashSet<String>();
    
    for (Iterator<MonitoredResource> it = monitoredConfs.values().iterator(); it.hasNext();) {
      MonitoredResource mr = it.next();
      if (mr.modified())
        result.add(mr.toString());
    }

    lastCheck = System.currentTimeMillis();

    return result;
  }

  public static void registerScannedRebelSource(RebelSource rs) {
    Object builder = getConf();

    if (builder == null) {
      log.error(new IllegalArgumentException("No builder provided"));
      return;
    }

    MonitoredResourceManager manager = getOrCreate(builder);
    if (!manager.rebelSources.contains(rs)) {
      manager.rebelSources.add(rs);
    }
  }

  public static Collection<RebelSource> getScannedRebelSource() {
    MonitoredResourceManager manager = get(getConf());

    if (manager == null) {
      return null;
    }
    return manager.rebelSources;
  }

  public static void registerScannedDir(File dir) {
    if (dir == null || !dir.isDirectory()) {
      return;
    }
    Object builder = getConf();

    if (builder == null) {
      log.error(new IllegalArgumentException("No builder provided"));
      return;
    }

    MonitoredResourceManager manager = getOrCreate(builder);
    manager.monitoredDirs.add(dir);
  }

  public static Collection<File> getScannedDir() {
    MonitoredResourceManager manager = get(getConf());

    if (manager == null) {
      return null;
    }
    return manager.monitoredDirs;
  }

  public static void registerConf(Class<?> klass) {
    registerConf(klass, getConf());
  }
  public static void registerConf(ClassLoader classLoader, String className) {
    registerConf(classLoader, className, getConf());
  }
  public static void registerConf(URL url) {
    registerConf(url, getConf());
  }
  public static void registerConf(String url) {
    registerConf(url, getConf());
  }
  
  public static void registerConf(Class<?> klass, Object builder) {
    if (klass == null)
      return;
    registerConf(toURL(klass), builder);
  }
  
  public static void registerConf(ClassLoader classLoader, String className, Object builder) {
    registerConf(toURL(classLoader, className), builder);
  }

  private static URL toURL(Class<?> klass) {
    URL url = classToURL.get(klass);
    if (url != null)
      return url;
    url = toURL(klass.getClassLoader(), klass.getName());
    if (url != null)
      classToURL.put(klass, url);
    return url;
  }

  private static URL toURL(ClassLoader classLoader, String className) {
    StopWatch sw = log.createStopWatch("MonitoredResourceManager#toURL");
    try {
      if (classLoader != null)
        return classLoader.getResource(className.replace('.', '/') + ".class");
      return null;
    }
    finally {
      sw.stop();
    }
  }

  public static void registerConf(String url, Object builder) {
    if (url == null)
      return;
    
    URL u;
    try {
      u = new URL(url);
    }
    catch (MalformedURLException e) {
      log.echoPrefix("Warning! Could not monitor configuration in '" + url + "'.");
      log.error(e);
      return;
    }
    registerConf(u, builder);
  }

  public static void registerConf(URL url, Object builder) {
    if (url == null || ResourceUtils.isJarURL(url) || "bundleresource".equals(url.getProtocol()))
      return;

    if (builder == null) {
      log.log("Warning! Could not monitor configuration in '" + url + "'.");
      log.error(new IllegalArgumentException("No builder provided"));
      return;
    }

    getOrCreate(builder).register(url);
  }

  /**
   * Find the MonitoredResourceManager instance that corresponds to that builder.
   * Checks if the url has already been registered to that instance.
   * 
   * @return if the builder has already registered this url.
   */
  public static boolean confRegistered(String url, Object builder) {
    if (url == null)
      return false;

    MonitoredResourceManager monitoredResourceManager = get(builder);
    if (monitoredResourceManager == null)
      return false;

    URL u;
    try {
      u = new URL(url);
    }
    catch (MalformedURLException e) {
      log.echoPrefix("Warning! Could not create url in '" + url + "'.");
      log.error(e);
      return false;
    }
    return monitoredResourceManager.monitoredConfs.containsKey(u);
  }

  private synchronized void register(URL url) {
    if (monitoredConfs.containsKey(url)) {
      return;
    }

    File file = ResourceUtils.getFile(url);
    if (!file.isFile() && !"jndi".equals(url.getProtocol())) {
      // expect writable resources to be in exploded deployments or rebel-mapped workspaces
      log.trace("Ignored non-file configuration in '{}'", url);
      return;
    }

    log.info("Monitoring configuration in '{}'", url);
    monitoredConfs.put(url, new MonitoredResource(ResourceUtil.asResource(url)));
  }

}
