package org.zeroturnaround.javarebel.integration.support;

import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.javarebel.ClassBytecodeProcessorCache;
import org.zeroturnaround.javarebel.IntegrationFactory;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.integration.util.JavassistUtil;

/**
 * Processors that extend this class are {@link #getCachingPolicy(ClassPool, ClassLoader) cached} by default.
 */
public abstract class CacheAwareJavassistClassBytecodeProcessor extends JavassistClassBytecodeProcessor {

  private static final Logger log = LoggerFactory.getLogger("CBP-Cache");
  private static final ClassBytecodeProcessorCache cbpCache = IntegrationFactory.getInstance().getClassBytecodeProcessorCache();

  @Override
  protected final byte[] process(ClassPool cp, ClassLoader cl, String classname, byte[] bytecode) throws Exception {
    CachingPolicy cachingPolicy = getCachingPolicy(cp, cl);
    String cacheKey = null;
    if (cachingPolicy != CachingPolicy.NEVER) {
      cacheKey = cbpCache.computeKey(classname, bytecode, this, cachingPolicy.getKey());
      if (cacheKey != null) {
        byte[] cachedBytes = cbpCache.get(cacheKey);
        if (cachedBytes != null) {
          log.trace("Found cached bytes for {} on {}", getClass().getName(), classname);
          return cachedBytes;
        }
      }
    }

    byte[] processedBytes = super.process(cp, cl, classname, bytecode);

    if (cacheKey != null) {
      log.trace("Caching transformation of {} on {}", getClass().getName(), classname);
      cbpCache.put(cacheKey, processedBytes);
    }

    return processedBytes;
  }

  /**
   * Returns the caching policy used by this processor.
   *
   * <p> If a processor uses a caching policy that isn't equal to {@link CachingPolicy#NEVER NEVER}, then its
   * transformations may be cached between runs. A processor then needs to make sure that its
   * {@link JavassistClassBytecodeProcessor#process(ClassPool, ClassLoader, CtClass) process} method does not have any
   * side effects. </p>
   *
   * <p> Also, the processing of a keyless processor may only depend on the structure of the processed class. In case of
   * any other dependencies (e.g. the existence or structure of other classes in the class pool, configuration options
   * etc), a {@link CachingPolicy#getKey() key} should be used to identify the processing configuration. This method
   * returns {@link CachingPolicy#ALWAYS ALWAYS} in its default implementation. </p>
   */
  protected CachingPolicy getCachingPolicy(ClassPool cp, ClassLoader cl) {
    return CachingPolicy.ALWAYS;
  }

  /**
   * A caching policy of a {@link JavassistClassBytecodeProcessor}. Every {@link JavassistClassBytecodeProcessor} may
   * choose to either never be cached or always be cached, optionally with a custom key.
   */
  public static class CachingPolicy {

    /**
     * A caching policy signifying that a processor can always be cached.
     */
    public static final CachingPolicy ALWAYS = new CachingPolicy(null);

    /**
     * A caching policy signifying that a processor must never be cached.
     */
    public static final CachingPolicy NEVER = new CachingPolicy(null);

    private final String key;

    private CachingPolicy(String key) {
      this.key = key;
    }

    /**
     * Returns the optional key used by this caching policy.
     *
     * @return the key used by this policy if any, or {@code null}
     */
    public String getKey() {
      return key;
    }

    public static CachingPolicy keyOnFlags(boolean flag) {
      return new CachingPolicy(flag ? "1" : "0");
    }

    public static CachingPolicy keyOnFlags(boolean... flags) {
      StringBuilder sb = new StringBuilder();
      for (boolean flag : flags) {
        sb.append(flag ? '1' : '0');
      }
      return new CachingPolicy(sb.toString());
    }

    public static CachingPolicy keyOnClasses(ClassPool cp, String... classNames) {
      boolean[] flags = new boolean[classNames.length];
      for (int i = 0; i < classNames.length; i++)
        flags[i] = JavassistUtil.hasClass(cp, classNames[i]);
      return keyOnFlags(flags);
    }

    public static CachingPolicy withKey(String key) {
      return new CachingPolicy(key);
    }
  }
}
