/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URL;

/**
 * Helper methods for managing streams
 */
public class IoUtil {

  private static final int BUFFER_SIZE = 4096;

  public static void write(byte[] bytes, OutputStream out) throws IOException {
    copy(new ByteArrayInputStream(bytes), out);
  }

  public static void copy(InputStream in, OutputStream out) throws IOException {
    byte[] buffer = new byte[BUFFER_SIZE];
    int read;

    while ((read = in.read(buffer)) != -1) {
      out.write(buffer, 0, read);
    }
  }

  public static void copy(Reader in, Writer out) throws IOException {
    char[] buffer = new char[BUFFER_SIZE];
    int read;

    while ((read = in.read(buffer)) != -1) {
      out.write(buffer, 0, read);
    }
  }

  public static byte[] getBytes(InputStream in) throws IOException {
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    copy(in, out);
    return out.toByteArray();
  }

  public static byte[] getBytesAndClose(Reader in) throws IOException {
    try {
      char[] charArray = new char[8 * 1024];
      StringBuilder builder = new StringBuilder();
      int numCharsRead;
      while ((numCharsRead = in.read(charArray, 0, charArray.length)) != -1) {
        builder.append(charArray, 0, numCharsRead);
      }
      return builder.toString().getBytes();
    }
    finally {
      try {
        in.close();
      }
      catch (IOException e) {
      }
    }
  }

  public static byte[] getBytesAndClose(InputStream in) {
    try {
      return getBytes(in);
    }
    catch (IOException e) {
      throw new RuntimeException(e);
    }
    finally {
      closeStream(in);
    }
  }

  public static void closeStream(Closeable in) {
    if (in != null) {
      try {
        in.close();
      }
      catch (Exception e) {
      }
    }
  }

  public static InputStream getInputStream(File file) {
    if (file.isDirectory())
      return null;

    try {
      return new BufferedInputStream(new FileInputStream(file));
    }
    catch (FileNotFoundException e) {
      throw new RuntimeException(e);
    }
  }

  public static URL getURL(File file) {
    try {
      return FileUtil.toURI(file).toURL();
    }
    catch (MalformedURLException e) {
      throw new RuntimeException(e);
    }
  }
}
