/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;

import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.Reloader;
import org.zeroturnaround.javarebel.ReloaderFactory;

/**
 * Helper methods for {@link Reloader}.
 * 
 * @author Rein Raudjärv
 * 
 * @see Reloader
 * @see ClassReflectionUtil
 */
public class ReloaderUtil {

  private static final Reloader reloader = ReloaderFactory.getInstance();
  private static final Method findLoadedClass = getFindLoadedClass();

  /**
   * Checks and reloads a set of classes.
   * @param classes a set of classes.
   * @see Reloader#checkAndReload(Class)
   */
  public static void checkAndReload(Collection<Class<?>> classes) {
    if (classes == null)
      return;
    
    for (Class<?> c : classes.toArray(new Class<?>[0]))
      reloader.checkAndReload(c);
  }

  public static void checkAndReloadHierarchy(Class<?> clazz) {
    checkAndReload(ClassReflectionUtil.getClassHierarchy(clazz));
  }

  // isReloadingClass
  
  /**
   * @param classes a set of classes.
   * @return <code>true</code> if any of the given classes is currently being reloaded.
   * @see Reloader#isReloadingClass(Class)
   */
  public static boolean isReloadingAnyClass(Collection<Class<?>> classes) {
    return isReloadingAnyClass(classes, null);
  }
  
  /**
   * @param classes a set of classes.
   * @param exclude a class to be excluded from the check
   * @return <code>true</code> if any of the given classes except the excluded class
   *    is currently being reloaded.
   * @see Reloader#isReloadingClass(Class)
   */
  public static boolean isReloadingAnyClass(Collection<Class<?>> classes, Class<?> exclude) {
    if (classes == null)
      return false;
    
    for (Class<?> klass : classes.toArray(new Class<?>[0])) {
      if ((exclude == null || !exclude.equals(klass))
          && reloader.isReloadingClass(klass))
        return true;
    }
    return false;
  }
  
  // isReloadableClass
  
  /**
   * @param classes a set of classes.
   * @return <code>true</code> if any of the given classes is reloadable.
   */
  public static boolean containsReloadableClass(Collection<Class<?>> classes) {
    if (classes == null)
      return false;
    
    for (Class<?> klass : classes) {
      if (reloader.isReloadableClass(klass))
        return true;
    }
    return false;
  }
  
  /**
   * @param classes a set of classes.
   * @return all the given classes that are reloadable.
   */
  public static Collection<Class<?>> getReloadableClasses(Collection<Class<?>> classes) {
    if (classes == null)
      return null;
    
    Collection<Class<?>> result = new ArrayList<Class<?>>();
    for (Class<?> klass : classes) {
      if (reloader.isReloadableClass(klass))
        result.add(klass);
    }
    return result;
  }
  
  /**
   * @param classes a set of classes.
   * @return the given set of classes where the not reloadable classes are removed.
   */
  private static Set<Class<?>> retainReloadableClasses(Set<Class<?>> classes) {
    for (Iterator<Class<?>> it = classes.iterator(); it.hasNext();) {
      Class<?> klass = it.next();
      if (!reloader.isReloadableClass(klass))
        it.remove();
    }
    return classes;
  }
  
  // Reloadable ClassHierarchy
  
  /**
   * @param klass a class.
   * @return <code>true</code> if any of the classes in the class hierarchy
   *   of the given class is reloadable.
   */
  public static boolean isReloadableClassHierarchy(Class<?> klass) {
    return containsReloadableClass(ClassReflectionUtil.getClassHierarchy(klass));
  }
  
  /**
   * @param classes a set class of classes.
   * @return <code>true</code> if any of the classes in the class hierarchies
   *   of the given classes is reloadable.
   */
  public static boolean existsReloadableClassHierarchy(Class<?>[] classes) {
    for (int i = 0; i < classes.length; i++) {
      Class<?> klass = classes[i];
      if (isReloadableClassHierarchy(klass))
        return true;
    }
    return false;
  }
  
  /**
   * @param klass a class.
   * @return the reloadable classes in the class hieracrhy of the given class.
   */
  public static Set<Class<?>> getReloadableClassHierarchy(Class<?> klass) {
    return retainReloadableClasses(ClassReflectionUtil.getClassHierarchy(klass));
  }
  
  /**
   * @param classes a set of classes.
   * @return the reloadable classes in the class hieracrhies of the given classes.
   */
  public static Set<Class<?>> getReloadableClassHierarchies(Class<?>[] classes) {
    return retainReloadableClasses(ClassReflectionUtil.getClassHierarchies(classes));
  }
  
  /**
   * @param classes a set of classes.
   * @return the reloadable classes in the class hieracrhies of the given classes.
   */
  public static Set<Class<?>> getReloadableClassHierarchies(Collection<Class<?>> classes) {
    return retainReloadableClasses(ClassReflectionUtil.getClassHierarchies(classes));
  }

  public static final Method getFindLoadedClass() {
    return ReflectionUtil.getDeclaredMethod(ClassLoader.class, "findLoadedClass", new Class[] { String.class });
  }

  public static final Class<?> getLoadedClass(ClassLoader cl, String name) {
    if (findLoadedClass == null)
      return null;
    try {
      return (Class<?>) findLoadedClass.invoke(cl, new Object[] { name });
    }
    catch (Exception e) {
      LoggerFactory.getLogger("Util").info("findLoadedClass for '" + name + "' failed on " + MiscUtil.identityToString(cl), e);
      return null;
    }
  }

  public static final boolean isLoadedClass(ClassLoader cl, String name) {
    return getLoadedClass(cl, name) != null;
  }

  // Deprecated

  /** @deprecated */
  public static Set<Class<?>> withParentInterfaces(Class<?> klass) {
    Set<Class<?>> result = ClassReflectionUtil.getClassHierarchies(klass.getInterfaces());
    result.add(klass);
    retainReloadableClasses(result);
    return result;
  }

  /** @deprecated */
  public static Set<Class<?>> withParentInterfaces(Collection<Class<?>> classes) {
    Set<Class<?>> result = ClassReflectionUtil.getClassHierarchies(ClassReflectionUtil.getInterfaces(classes));
    result.addAll(classes);
    retainReloadableClasses(result);
    return result;
  }

}
