/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.lang.ref.WeakReference;

import org.zeroturnaround.javarebel.RequestIntegrationFactory;
import org.zeroturnaround.javarebel.RequestListener;

public class RequestListenerUtil {

  /**
   * The target listener will be weakly referenced
   * 
   * @param listener
   * @return listener that processes events with current context class loader  
   */
  public static RequestListener bindContextClassLoader(RequestListener listener) {
    return bindClassLoader(listener, MiscUtil.getContextClassLoader());
  }

  /**
   * The target listener will be weakly referenced
   * 
   * @param listener
   * @param cl
   * @return listener that processes events with given class loader context class loader
   */
  public static RequestListener bindClassLoader(RequestListener listener, ClassLoader cl) {
    if (!canSetTCL() || cl == null)
      return WeakUtil.weak(listener);
    return new BoundRequestListener(listener, cl);
  }

  private static class BoundRequestListener implements RequestListener, WeakUtil.RemovableListener {
    private final WeakReference<RequestListener> targetRef;
    private final WeakReference<ClassLoader> classLoderRef;
    private final String identity;
    private final int priority;

    public BoundRequestListener(RequestListener target, ClassLoader cl) {
      this.targetRef = new WeakUtil.WeakListenerReference<RequestListener>(this, target);
      this.classLoderRef = new WeakReference<ClassLoader>(cl);
      this.identity = "boundCL(" + MiscUtil.identityToString(cl) + ")[" + MiscUtil.dumpToString(target) + "]";
      this.priority = target.priority();
    }

    private ClassLoader getClassLoader() {
      return classLoderRef.get();
    }

    private RequestListener getTarget() {
      return targetRef.get();
    }

    public boolean rawRequest(Object context, Object request, Object response) throws Exception {
      RequestListener target = getTarget();
      if (target == null)
        return false;

      ClassLoader cl = getClassLoader();
      boolean hasCl = cl != null;
      ClassLoader old = hasCl ? MiscUtil.setContextClassLoader(cl) : null;
      try {
        return target.rawRequest(context, request, response);
      }
      finally {
        if (hasCl) MiscUtil.setContextClassLoader(old);
      }
    }

    public void beforeRequest() throws Exception {
      RequestListener target = getTarget();
      if (target == null)
        return;

      ClassLoader cl = getClassLoader();
      boolean hasCl = cl != null;
      ClassLoader old = hasCl ? MiscUtil.setContextClassLoader(cl) : null;
      try {
        target.beforeRequest();
      }
      finally {
        if (hasCl) MiscUtil.setContextClassLoader(old);
      }
    }

    public void requestFinally() throws Exception {
      RequestListener target = getTarget();
      if (target == null)
        return;

      ClassLoader cl = getClassLoader();
      boolean hasCl = cl != null;
      ClassLoader old = hasCl ? MiscUtil.setContextClassLoader(cl) : null;
      try {
        target.requestFinally();
      }
      finally {
        if (hasCl) MiscUtil.setContextClassLoader(old);
      }
    }

    public int priority() {
      return priority;
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      RequestIntegrationFactory.getInstance().removeRequestListener(this);
    }
  }

  private static boolean canSetTCL() {
    // oc4j9 server thread overrides getContextClassLoader
    return !"com.evermind.server.ApplicationServerThread".equals(Thread.currentThread().getClass().getName());
  }

}
