/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.util.ArrayDeque;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.Set;

/**
 * Class reflection helper methods.
 * 
 * @author Rein Raudjärv
 */
public abstract class ClassReflectionUtil {
  
  /**
   * @param klass a root class.
   * @return a set containing the given class, the super classes,
   *   the interfaces and all the parent interfaces. 
   */
  public static Set<Class<?>> getClassHierarchy(Class<?> klass) {
    return getClassHierarchies(Collections.singleton(klass));
  }
  
  /**
   * @param classes root classes.
   * @return a set containing the given classes, the super classes,
   *         the interfaces and all the parent interfaces.
   */
  public static Set<Class<?>> getClassHierarchies(Class<?>... classes) {
    return getClassHierarchies(Arrays.asList(classes));
  }
  
  /**
   * @param classes root classes.
   * @return a set containing the given classes, the super classes,
   *   the interfaces and all the parent interfaces. 
   */  
  public static Set<Class<?>> getClassHierarchies(Collection<? extends Class<?>> classes) {
    // A stack containing the classes and interface to be read
    Deque<Class<?>> stack = new ArrayDeque<Class<?>>(classes);

    // Result set
    Set<Class<?>> result = new HashSet<Class<?>>();

    // Continue until the stack is empty
    while (!stack.isEmpty()) {
      // Remove the first item from the stack
      Class<?> klass = stack.removeFirst();

      // Add it to the results
      result.add(klass);
      
      // Find super class
      Class<?> superClass = klass.getSuperclass();
      if (superClass != null && !result.contains(superClass))
        stack.add(superClass);

      // Find interfaces
      Class<?>[] interfaces = klass.getInterfaces();
      if (interfaces != null) {
        // For each parent interface
        for (int i = 0; i < interfaces.length; i++) {
          Class<?> iface = interfaces[i];
          // Add it to the stack unless it's in the results
          // This ensures that each interface is only processed once 
          if (!result.contains(iface))
            stack.add(iface);
        }
      }
    }
    
    return result;
  }
  
  /**
   * @param classes a set of classes.
   * @return a set containing interfaces of each class.
   */
  public static Set<Class<?>> getInterfaces(Collection<Class<?>> classes) {
    if (classes == null)
      return null;
    Set<Class<?>> result = new HashSet<Class<?>>();
    for (Class<?> klass : classes) {
      if (klass != null) {
        result.addAll(Arrays.asList((Class<?>[]) klass.getInterfaces()));
      }
    }
    return result;
  }
}
