/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;

import org.zeroturnaround.javarebel.ClassResourceSource;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.Resource;
import org.zeroturnaround.javarebel.ResourceSource;
import org.zeroturnaround.javarebel.integration.support.BaseClassResourceSource;
import org.zeroturnaround.javarebel.support.FileResource;
import org.zeroturnaround.javarebel.support.ResourceUtils;
import org.zeroturnaround.javarebel.support.URLResource;

/**
 * Helper methods for finding resources.
 * 
 * @author Rein Raudjärv
 * 
 * @see Resource
 * @see ResourceSource
 * @see ClassLoader
 * @see URLClassLoader
 * @see URL
 * @see InputStream
 */
public class ResourceUtil {

  private static final Method FIND_RESOURCE_METHOD = getClassLoaderMethod("findResource", String.class);
  private static final Method FIND_RESOURCES_METHOD = getClassLoaderMethod("findResources", String.class);

  /**
   * Returns a <code>Method</code> object that reflects the specified
   * declared method of the {@link ClassLoader}.
   * 
   * @param methodName the name of the method.
   * @param paramTypes the parameter array.
   * @return the <code>Method</code> object for the method of the {@link ClassLoader}
   *         matching the specified name and parameters.
   */
  private static Method getClassLoaderMethod(String methodName, Class<?>... paramTypes) {
    Method result = ReflectionUtil.getDeclaredMethod(ClassLoader.class, methodName, paramTypes);
    if (result == null) {
      LoggerFactory.getLogger("Util").errorEcho("Could not find {} in ClassLoader", methodName);
    }
    return result;
  }

  /**
   * Invokes the <code>findResource</code> method of the given class loader.
   * 
   * <p>
   * In case of {@link URLClassLoader} the underlying public method is invoked directly.
   * Otherwise the protected method of the {@link ClassLoader} is invoked through the
   * reflection.
   * </p>
   * 
   * @param classloader the class loader object.
   * @param name the name of the resource
   *          (parameter passed to the underlying <code>findResource</code> method)
   * @return a <code>URL</code> for the resource, or <code>null</code>
   *         if the resource could not be found.
   * 
   * @see URLClassLoader#findResource(String)
   */
  public static URL findResource(ClassLoader classloader, String name) {
    if (name == null) {
      return null;
    }

    URL url;
    if (classloader instanceof URLClassLoader)
      url = ((URLClassLoader) classloader).findResource(name);
    else {
      try {
        url = (URL) FIND_RESOURCE_METHOD.invoke(classloader, new Object[] { name });
      }
      catch (IllegalArgumentException e) {
        throw new RuntimeException(e);
      }
      catch (IllegalAccessException e) {
        throw new RuntimeException(e);
      }
      catch (InvocationTargetException e) {
        throw new RuntimeException(e.getTargetException());
      }
    }
    return url;
  }

  /**
   * Invokes the <code>findResources</code> method of the given class loader.
   * 
   * <p>
   * In case of {@link URLClassLoader} the underlying public method is invoked directly.
   * Otherwise the protected method of the {@link ClassLoader} is invoked through the
   * reflection.
   * </p>
   * 
   * <p>
   * The underlying <code>Enumeration</code> is converted to an array of <code>URL</code>s.
   * 
   * @param classloader the class loader object.
   * @param name the resource name.
   *          (parameter passed to the underlying <code>findResources</code> method)
   * @return <code>URL</code>s for the resources, or <code>null</code>
   *         if no resource could be found.
   * 
   * @see URLClassLoader#findResources(String)
   * @see #findResources(URLClassLoader, String)
   * @see #findResource(ClassLoader, String)
   * @see #toURLs(Enumeration)
   */
  @SuppressWarnings("unchecked")
  public static URL[] findResources(ClassLoader classloader, String name) {
    Enumeration<URL> en;
    if (classloader instanceof URLClassLoader)
      try {
        en = ((URLClassLoader) classloader).findResources(name);
      }
      catch (IOException e) {
        throw new RuntimeException(e);
      }
    else {
      try {
        en = (Enumeration<URL>) FIND_RESOURCES_METHOD.invoke(classloader, new Object[] { name });
      }
      catch (IllegalArgumentException e) {
        throw new RuntimeException(e);
      }
      catch (IllegalAccessException e) {
        throw new RuntimeException(e);
      }
      catch (InvocationTargetException e) {
        throw new RuntimeException(e.getTargetException());
      }
    }

    if (en == null || !en.hasMoreElements())
      return null;

    return toURLs(en);
  }

  /**
   * Invokes the <code>findResources</code> method of the given URL class loader.
   * 
   * <p>
   * The underlying public method is invoked directly and
   * the <code>Enumeration</code> is converted to an array of <code>URL</code>s.
   * 
   * @param classloader the class loader object.
   * @param name the resource name.
   *          (parameter passed to the underlying <code>findResources</code> method)
   * @return <code>URL</code>s for the resources, or <code>null</code>
   *         if no resource could be found.
   * 
   * @see URLClassLoader#findResources(String)
   * @see #findResources(ClassLoader, String)
   * @see #findResource(ClassLoader, String)
   * @see #toURLs(Enumeration)
   */
  public static URL[] findResources(URLClassLoader classloader, String name) {
    Enumeration<URL> en;
    try {
      en = classloader.findResources(name);
    }
    catch (IOException e) {
      throw new RuntimeException(e);
    }

    if (en == null || !en.hasMoreElements())
      return null;

    return toURLs(en);
  }

  /**
   * Converts the <code>Enumeration</code> to an array of <code>URL</code>s.
   * 
   * @param en an <code>Enumeration</code> of <code>URL</code>s (not <code>null</code>).
   * @return an array of <code>URL</code>s.
   * 
   * @see #toEnumeration(URL[])
   */
  public static URL[] toURLs(Enumeration<URL> en) {
    if (en == null)
      return null;

    List<URL> urls = new ArrayList<URL>();
    while (en.hasMoreElements())
      urls.add(en.nextElement());

    return urls.toArray(new URL[urls.size()]);
  }

  /**
   * Converts the array of <code>URL</code>s to an <code>Enumeration</code>.
   * 
   * @param urls an array of <code>URL</code>s (not <code>null</code>).
   * @return an <code>Enumeration</code> of <code>URL</code>s
   * 
   * @see #toURLs(Enumeration)
   */
  public static Enumeration<URL> toEnumeration(URL[] urls) {
    return Collections.enumeration(Arrays.asList(urls));
  }

  public static Enumeration<URL> toEnumeration(File f) {
    return Collections.enumeration(Arrays.asList(IoUtil.getURL(f)));
  }

  /**
   * Concatenates two arrays of <code>URL</code>s into one.
   * 
   * @param first first array of <code>URL</code>s (maybe <code>null</code>)
   * @param second second array of <code>URL</code>s (maybe <code>null</code>)
   * @return array of <code>URL</code>s containing all elements (maybe <code>null</code>).
   */
  public static URL[] concat(URL[] first, URL[] second) {
    if (first == null)
      return second;
    if (second == null)
      return first;

    List<URL> list = new ArrayList<URL>();
    list.addAll(Arrays.asList(first));
    list.addAll(Arrays.asList(second));
    return list.toArray(new URL[list.size()]);
  }

  /**
   * Concatenates two class paths.
   * 
   * @param first first class path.
   * @param second second class path.
   * @return result containing both class path entries.
   */
  public static String concatClassPath(String first, String second) {
    if (first == null)
      return second;
    if (second == null)
      return first;

    return first + File.pathSeparator + second;
  }

  /**
   * Converts an array of <code>URL</code>s into a class path.
   * <p>
   * <code>URL</code> will be separated by {@link File#pathSeparator}s.
   * 
   * @param urls array of <code>URL</code>s (maybe <code>null</code>)
   * @return class path.
   */
  public static String toClassPath(URL[] urls) {
    if (urls == null)
      return null;

    StringBuilder sb = new StringBuilder();
    for (int i = 0; i < urls.length; i++) {
      URL url = urls[i];
      File file = ResourceUtils.getFile(url);
      sb.append(file.getPath());
      if (i < urls.length - 1)
        sb.append(File.pathSeparator);
    }
    return sb.toString();
  }

  /**
   * Finds a class resource using a resource finder.
   * 
   * <p>
   * This is used in {@link BaseClassResourceSource}
   * which is the base implementation of {@link ClassResourceSource}.
   * </p>
   * 
   * <p>
   * The class name is converted to a resource name
   * by replacing all dots with slashes
   * and adding <code>.class</code> to the end.
   * </p>
   * 
   * @param source the resource finder.
   * @param className the name of the class.
   * @return {@link Resource} that corresponds to the given class.
   * 
   * @see ResourceSource
   * @see ClassResourceSource
   * @see BaseClassResourceSource
   */
  public static Resource getClassResource(ResourceSource source, String className) {
    return source.getLocalResource(className.replace('.', '/') + ".class");
  }

  /**
   * Creates a one-element array of <code>Resource</code>s.
   * 
   * @param resource the <code>Resource</code> element.
   * @return a one-element array with the given element
   *         or <code>null</code> if the element was <code>null</code>.
   */
  public static Resource[] asArray(Resource resource) {
    if (resource == null)
      return null;
    return new Resource[] { resource };
  }

  /**
   * Creates a new <code>Resource</code> object based on the given <code>URL</code>.
   * 
   * <p>
   * If a <code>file</code> or <code>vfsfile</code> protocol is used
   * a {@link FileResource} will be created.
   * Otherwise a new {@link URLResource} instance will be returned.
   * 
   * @param url the resource <code>URL</code>.
   * @return new <code>Resource</code> object based on the given <code>URL</code>.
   */
  public static Resource asResource(URL url) {
    if (url == null)
      return null;

    if (ResourceUtils.isFileURL(url))
      return new FileResource(url);
    else
      return new URLResource(url);
  }

  /**
   * Creates an array of <code>Resource</code> objects
   * based on the given <code>URL</code> objects.
   * 
   * @param urls the resource <code>URL</code> objects.
   * @return new <code>Resource</code> objects.
   */
  public static Resource[] asResources(URL[] urls) {
    if (urls == null)
      return null;

    Resource[] result = new Resource[urls.length];
    for (int i = 0; i < urls.length; i++)
      result[i] = asResource(urls[i]);
    return result;
  }

  /**
   * Creates an array of <code>Resource</code> objects
   * based on the given <code>URL</code> objects.
   * 
   * @param urls the resource <code>URL</code> objects.
   * @return new <code>Resource</code> objects.
   */
  public static Resource[] asResources(Enumeration<URL> urls) {
    return asResources(toURLs(urls));
  }

  /**
   * Creates an array of <code>Resource</code> objects
   * based on the given <code>URL</code> objects.
   * 
   * @param urls the resource <code>URL</code> objects.
   * @return new <code>Resource</code> objects.
   */
  public static Resource[] asResources(List<URL> urls) {
    if (urls == null)
      return null;

    Resource[] result = new Resource[urls.size()];
    int i = 0;
    for (URL u : urls)
      result[i++] = asResource(u);
    return result;
  }

  /**
   * Creates a raw input stream from the given URL.
   * 
   * @param url <code>URL</code> object.
   * @return a raw input stream.
   * 
   * @see #asInputStream(URL)
   */
  public static InputStream asRawInputStream(URL url) {
    try {
      if (url != null)
        return url.openStream();
    }
    catch (IOException e) {
    }
    return null;
  }

  /**
   * Creates a byte input stream from the given URL.
   * 
   * @param url <code>URL</code> object.
   * @return a byte input stream.
   * 
   * @see #asRawInputStream(URL)
   */
  public static InputStream asInputStream(URL url) {
    if (url == null)
      return null;

    byte[] bytes = asResource(url).getBytes();
    if (bytes == null)
      return null;

    return new ByteArrayInputStream(bytes);
  }

  /**
   * Converts a {@link File} into a {@link URL}.
   * 
   * @param file <code>File</code> object.
   * @return <code>URL</code> object.
   */
  public static URL makeURL(File file) throws MalformedURLException {
    if (file == null)
      return null;

    return FileUtil.toURI(file).toURL();
  }

  /**
   * Converts a JAR entry into a <code>URL</code>.
   * 
   * @param jar a JAR file.
   * @param entry a JAR entry name.
   * 
   * @return <code>URL</code> object.
   */
  public static URL makeURL(File jar, String entry) throws MalformedURLException {
    if (jar == null)
      return null;
    if (entry == null)
      return makeURL(jar);

    return new URL("jar:" + FileUtil.toURI(jar).toURL() + "!/" + entry);
  }

  /**
   * Converts a file path or URL into a <code>File</code>.
   * 
   * @param path file path or URL.
   * @return <code>File</code> object.
   */
  public static File getFileFromPathOrURL(String path) throws MalformedURLException {
    if (path == null)
      return null;

    File result = new File(path);
    if (result.exists())
      return result;

    return ResourceUtils.getFile(new URL(path));
  }

  /**
   * Converts a URL into a <code>File</code>.
   * 
   * @param url URL.
   * @return <code>File</code> object.
   */
  public static File getFileFromURL(String url) throws MalformedURLException {
    if (url == null)
      return null;

    return ResourceUtils.getFile(new URL(url));
  }

  /**
   * Converts a URL into a <code>File</code>.
   * 
   * @param url URL.
   * @return <code>File</code> object.
   */
  public static File getFileFromURL(URL url) {
    if (url == null)
      return null;

    return ResourceUtils.getFile(url);
  }

}