/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.generic;

import java.net.URL;
import java.security.AccessControlContext;
import java.security.AccessController;
import java.security.PrivilegedAction;
import org.zeroturnaround.javarebel.Resource;

/**
 * @author Rein Raudjärv
 * 
 * @see RestrictedClassResourceSource
 */
public final class RestrictedResource implements Resource {
  
  public static Resource newInstance(AccessControlContext acc, Resource target) {
    if (acc == null || target == null)
      return null;
    
    return new RestrictedResource(acc, target);
  }
  
  private final AccessControlContext acc;
  private final Resource target;
  
  private RestrictedResource(AccessControlContext acc, Resource target) {
    this.acc = acc;
    this.target = target;
  }
  
  public Resource getTarget() {
    return target;
  }

  public byte[] getBytes() {
    if (System.getSecurityManager() == null)
      return target.getBytes();

    return AccessController.doPrivileged(new PrivilegedAction<byte[]>() {
      public byte[] run() {
        return target.getBytes();
      }
    }, acc);
  }

  public long lastModified() {
    if (System.getSecurityManager() == null)
      return target.lastModified();

    Long obj = AccessController.doPrivileged(new PrivilegedAction<Long>() {
      public Long run() {
        return target.lastModified();
      }
    }, acc);
    return obj.longValue();
  }

  public URL toURL() {
    if (System.getSecurityManager() == null)
      return target.toURL();

    return AccessController.doPrivileged(new PrivilegedAction<URL>() {
      public URL run() {
        return target.toURL();
      }
    }, acc);
  }
  
  public String toString() {
    return "Privileged(" + targetToString() + ")";
  }
  
  private String targetToString() {
    if (System.getSecurityManager() == null)
      return target.toString();

    return AccessController.doPrivileged(new PrivilegedAction<String>() {
      public String run() {
        return target.toString();
      }
    });
  }

}
