package org.zeroturnaround.javarebel.integration.util;

import java.lang.reflect.AccessibleObject;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.ReloaderFactory;

public class ReflectionUtil {

  /**
   * Invoke a declared method with three parameters.
   */
  @SuppressWarnings("unchecked")
  public static <T> T invoke(Object target, String methodName, Object param1, Object param2, Object param3) {
    return (T) invokeByArgs(target, methodName, param1, param2, param3);
  }

  /**
   * Invoke a declared method with two parameters.
   */
  @SuppressWarnings("unchecked")
  public static <T> T invoke(Object target, String methodName, Object param1, Object param2) {
    return (T) invokeByArgs(target, methodName, param1, param2);
  }

  /**
   * Invoke a declared method with one parameter.
   */
  @SuppressWarnings("unchecked")
  public static <T> T invoke(Object target, String methodName, Object param1) {
    return (T) invokeByArgs(target, methodName, param1);
  }

  /**
   * Invoke a declared method with no parameters
   */
  @SuppressWarnings("unchecked")
  public static <T> T invoke(Object target, String methodName) {
    return (T) invokeByArgs(target, methodName);
  }

  public static Object invokeByArgs(Object target, String methodName, Object... args) {
    try {
      Method method = findDeclaredMethod(target.getClass(), methodName, args);
      setAccessible(method);
      return method.invoke(target, args);
    }
    catch (Exception e) {
      throw new RuntimeException(e);
    }
  }
  
  public static Object invokeStaticByArgs(Class<?> klass, String methodName, Object... args) {
    try {
      Method method = findDeclaredMethod(klass, methodName, args);
      setAccessible(method);
      return method.invoke(null, args);
    }
    catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Find the single declared method in the class hierarchy that matches the name and parameters.
   */
  public static Method findDeclaredMethod(Class<?> klass, String methodName, Object... args) throws NoSuchMethodException {
    List<Method> result = new ArrayList<Method>();
    Class<?> curr = klass;
    Set<String> nonPrivateSignatures = new HashSet<String>();
    while (curr != null) {
      for (Method method : curr.getDeclaredMethods()) {
        if (method.getName().equals(methodName) && acceptsParameters(method, args)) {
          if (Modifier.isPrivate(method.getModifiers()) || nonPrivateSignatures.add(sig(method))) {
            result.add(method);
          }
        }
      }
      curr = curr.getSuperclass();
    }
    if (result.size() > 1) {
      throw new NoSuchMethodException("ambiguous match: " + methodName + " on class " + klass.getName());// NOSONAR
    }
    if (result.isEmpty()) {
      throw new NoSuchMethodException("no such method: " + methodName + " on class " + klass.getName());// NOSONAR
    }
    return result.get(0);
  }

  private static String sig(Method method) {
    return method.getName() + MiscUtil.toString(method.getParameterTypes());
  }

  private static boolean acceptsParameters(Method method, Object[] args) {
    Class<?>[] parameterTypes = method.getParameterTypes();
    if (parameterTypes.length != args.length) {
      return false;
    }
    for (int i = 0; i < parameterTypes.length; i++) {
      if (args[i] != null && !parameterTypes[i].isInstance(args[i]) && !isBoxed(parameterTypes[i], args[i]))
        return false;
    }
    return true;
  }

  private static boolean isBoxed(Class<?> expectedType, Object arg) {
    if (boolean.class.equals(expectedType))
      return arg instanceof Boolean;
    if (byte.class.equals(expectedType))
      return arg instanceof Byte;
    if (char.class.equals(expectedType))
      return arg instanceof Character;
    if (short.class.equals(expectedType))
      return arg instanceof Short;
    if (int.class.equals(expectedType))
      return arg instanceof Integer;
    if (long.class.equals(expectedType))
      return arg instanceof Long;
    if (float.class.equals(expectedType))
      return arg instanceof Float;
    if (double.class.equals(expectedType))
      return arg instanceof Double;
    return false;
  }

  public static <T> T getFieldValue(final Object o, String field) {
    try {
      return getFieldValue(o.getClass(), o, field);
    }
    catch (Exception e) {
      LoggerFactory.getLogger("ReflectionUtil").error("Searching for " + MiscUtil.identityToString(o) + "." + field + " but got:" + e.getMessage(), e);
    }
    return null;
  }

  public static <T> T getFieldValue(Class<?> orig, String field) {
    try {
      return getFieldValue(orig, null, field);
    }
    catch (Exception e) {
      LoggerFactory.getLogger("ReflectionUtil").error("Searching for " + orig.getName() + "." + field + " but got:" + e.getMessage(), e);// NOSONAR
    }
    return null;
  }

  @SuppressWarnings("unchecked")
  private static <T> T getFieldValue(Class<?> orig, Object o, String field) throws IllegalArgumentException, IllegalAccessException {
    Class<?> k = orig;
    while (k != null) {
      try {
        Field f = k.getDeclaredField(field);
        setAccessible(f);
        return (T) f.get(o);
      }
      catch (NoSuchFieldException e) {
      }
      k = k.getSuperclass();
    }
    return null;
  }

  public static void setFieldValue(Object instance, String field, Object value) {
    setFieldValue(instance, instance.getClass(), field, value);
  }

  public static void setFieldValueOrFail(Object instance, String field, Object value) throws Exception {
    setFieldValueOrFail(instance, instance.getClass(), field, value);
  }

  public static void setFieldValue(final Object instance, Class<?> klass, String field, Object value) {
    try {
      setFieldValueOrFail(instance, klass, field, value);
    }
    catch (Exception e) {
      LoggerFactory.getLogger("ReflectionUtil").error("Searching for " + MiscUtil.identityToString(instance) + "." + field + " but got:" + e.getMessage(), e);
    }
  }

  /**
   * @param instance the instance
   * @param klass lowest class in hierarchy to look into, restriction is needed when otherwise you might detect the same field name from unknown subclasses and change this instead
   * @param field the field name
   * @param value the new value of the field
   * @throws Exception if field not set
   */
  public static void setFieldValueOrFail(final Object instance, Class<?> klass, String field, Object value) throws Exception {
    if (klass == null) {
      throw new IllegalArgumentException("klass == null");
    }
    Exception exception = null;
    while (klass != null) {
      try {
        Field f = klass.getDeclaredField(field);
        setAccessible(f);
        f.set(instance, value);
        return;
      } catch (NoSuchFieldException e) {
        exception = e;
      }
      klass = klass.getSuperclass();
    }
    throw exception;
  }

  public static void setAccessible(final AccessibleObject m) {
    if (System.getSecurityManager() == null) {
      m.setAccessible(true);
    }
    else {
      SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<Object>() {
        public Object run() {
          m.setAccessible(true);
          return null;
        }
      });
    }
  }

  public static Field getDeclaredField(Class<?> klass, String name) {
    try {
      final Field f = klass.getDeclaredField(name);
      setAccessible(f);
      return f;
    }
    catch (NoClassDefFoundError e) {
    }
    catch (SecurityException e) {
    }
    catch (NoSuchFieldException e) {
    }
    return null;
  }

  public static Method getDeclaredMethod(Class<?> klass, String method, Class<?>... paramTypes) {
    try {
      Method m = klass.getDeclaredMethod(method, paramTypes);
      setAccessible(m);
      return m;
    }
    catch (SecurityException e) {
    }
    catch (NoSuchMethodException e) {
    }
    return null;
  }

  public static Constructor<?> getDeclaredConstructor(Class<?> klass, Class<?>... paramTypes) {
    try {
      return klass.getDeclaredConstructor(paramTypes);
    }
    catch (SecurityException e) {
    }
    catch (NoSuchMethodException e) {
    }
    return null;
  }

  public static Class<?> getClassForAnyName(ClassLoader loader, String... names) {
    for (String name : names) {
      try {
        return loader.loadClass(name);
      }
      catch (ClassNotFoundException ignored) {
      }
    }
    throw new RuntimeException("None of the following classes were found: " + Arrays.asList(names));
  }

  public static void copyDeclaredFields(Class<?> klass, Object src, Object dest) {
    if (!klass.isInstance(src)) {
      throw new RuntimeException("src should be instance of klass");
    }
    if (!klass.isInstance(dest)) {
      throw new RuntimeException("dest should be instance of klass");
    }

    Field[] fields = klass.getDeclaredFields();
    for (Field f : fields) {
      if (Modifier.isStatic(f.getModifiers()) || Modifier.isFinal(f.getModifiers()) && !ReloaderFactory.getInstance().isReloadableClass(klass))
        continue;

      setAccessible(f);
      try {
        f.set(dest, f.get(src));
      }
      catch (IllegalArgumentException e) {
        throw new RuntimeException(e);
      }
      catch (IllegalAccessException e) {
        throw new RuntimeException(e);
      }
    }
  }
}