package org.zeroturnaround.jrebel.liferay;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.zeroturnaround.javarebel.ConfigurationFactory;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.ReloaderFactory;
import org.zeroturnaround.javarebel.RequestIntegrationFactory;
import org.zeroturnaround.javarebel.integration.generic.RequestListenerAdapter;
import org.zeroturnaround.javarebel.integration.monitor.MonitoredResource;
import org.zeroturnaround.javarebel.integration.util.ReflectionUtil;
import org.zeroturnaround.javarebel.integration.util.RequestListenerUtil;
import org.zeroturnaround.javarebel.integration.util.ResourceUtil;
import org.zeroturnaround.jrebel.liferay.util.JrPropsUtil;
import org.zeroturnaround.jrebel.liferay.util.PropertiesWrapper;

import com.liferay.portal.configuration.ConfigurationImpl;
import com.liferay.portal.kernel.configuration.Configuration;
import com.liferay.portal.util.PropsFiles;
import com.liferay.portal.util.PropsUtil;
import com.liferay.portal.util.PropsValues;


public class PropsUtilReloader extends RequestListenerAdapter {
  private static final Logger log = LoggerFactory.getLogger(LiferayPlugin.PRODUCT_PREFIX);
  private static final int CHECK_INTERVAL = ConfigurationFactory.getInstance().getCheckInterval();
  
  private final JrPropsUtil propsUtil;
  private final Map<String, MonitoredResource> commonResources = new HashMap<String, MonitoredResource>();
  private final Map<String, MonitoredResource> companyResources = new HashMap<String, MonitoredResource>();
  
  private final List<PropertiesWrapper> propertiesWrappers = new ArrayList<PropertiesWrapper>();
  private final Map<Object, PropertiesWrapper> companyPropertiesWrappers = new HashMap<Object, PropertiesWrapper>();
  
  private volatile long lastCheck;
  
  public static PropsUtilReloader createStaticReloader() {
    return StaticReloaderHolder.INSTANCE;
  }
  
  public PropsUtilReloader(JrPropsUtil propsUtil) {
    this.propsUtil = propsUtil;
  }
  
  public void start() {
    RequestIntegrationFactory.getInstance().addRequestListener(RequestListenerUtil.bindContextClassLoader(this));
  }

  public void beforeRequest() {
    if (lastCheck + CHECK_INTERVAL > System.currentTimeMillis()) {
      return;
    }
    
    checkCompanyResources();
    checkCommonResources();
    
    lastCheck = System.currentTimeMillis();
  }

  private void checkCommonResources() {
    boolean modified = false;
    for (MonitoredResource resource : commonResources.values()) {
      if (resource.modified()) {
        log.info("Resource modified : {}", resource);
        modified = true;
      }
    }
    if (modified) {
      propsUtil.jrClearProperties();
      ReloaderFactory.getInstance().reinitClass(PropsValues.class);
      for (PropertiesWrapper wrapper : propertiesWrappers) {
        ConfigurationImpl ci = propsUtil.jrNewConfigurationImpl(wrapper.getClassLoader(), wrapper.getName());
        if (ci == null)
          continue;
        Properties properties = ci.getProperties();
        propsUtil.jrAddProperties(properties, null);
      }
    }
  }

  private void checkCompanyResources() {
    boolean modified = false;
    for (MonitoredResource resource : companyResources.values()) {
      if (resource.modified()) {
        log.info("Company resource modified : {}", resource);
        modified = true;
        break; // will clear all MonitoredResources anyway, no need to continue
      }
    }
    if (modified) {
      companyResources.clear();
      propsUtil.jrClearCompanyProperties();
      for(Map.Entry<Object, PropertiesWrapper> e : companyPropertiesWrappers.entrySet()) {
        ConfigurationImpl ci = propsUtil.jrNewConfigurationImpl(e.getValue().getClassLoader(), e.getValue().getName());
        if (ci == null)
          continue;
        Properties properties = ci.getProperties();
        propsUtil.jrAddProperties(properties, e.getKey());
      }
    }
  }
  
  public void registerSource(List<String> sources) {
    registerMonitoredResources(sources, commonResources);
  }
  
  public void registerCompanySources(List<String> sources) {
    registerMonitoredResources(sources, companyResources);
  }
  
  public void registerExternalSource(Properties properties) {
    if (properties instanceof PropertiesWrapper) {
      PropertiesWrapper wrapper = (PropertiesWrapper) properties;
      MonitoredResource mr = new MonitoredResource(ResourceUtil.asResource(wrapper.getResource()));
      commonResources.put(mr.toString(), mr);
      log.info("Watching resource {}", wrapper.getResource());
      propertiesWrappers.add(wrapper);
    }
  }
  
  public void registerExternalSource(Properties properties, Object company) {
    if (properties instanceof PropertiesWrapper) {
      PropertiesWrapper wrapper = (PropertiesWrapper) properties;
      MonitoredResource mr = new MonitoredResource(ResourceUtil.asResource(wrapper.getResource()));
      commonResources.put(mr.toString(), mr);
      log.info("Watching company resource {}", wrapper.getResource());
      
      companyPropertiesWrappers.put(company, wrapper);
    }
  }
  
  private void registerMonitoredResources(List<String> sources, Map<String, MonitoredResource> monitoredResources) {
    for (String source : sources) {
      log.info("Monitored: {}", source);
      try {
        MonitoredResource mr = new MonitoredResource(ResourceUtil.asResource(new URL(source)));
        monitoredResources.put(mr.toString(), mr);
      }
      catch (MalformedURLException e) {
        log.error("{} isn't a valid URL", source);
      }
    }
  }
  
  private static class StaticReloaderHolder {
    static final PropsUtilReloader INSTANCE = initPropsUtilInstance();

    private static PropsUtilReloader initPropsUtilInstance() {
      ReflectionJrPropsUtil jrPropsUtil = new ReflectionJrPropsUtil();
      PropsUtilReloader reloader = new PropsUtilReloader(jrPropsUtil);
      reloader.start();
      return reloader;
    }
  }
  
  private static class ReflectionJrPropsUtil implements JrPropsUtil {
    
    private final Class<?> propsUtilClass;
    private final Field configurationField;
    private final Field configurationsField;
    private final Constructor<?> configurationConstructor;
    
    public ReflectionJrPropsUtil() {
      this.propsUtilClass = PropsUtil.class;
      this.configurationField = ReflectionUtil.getDeclaredField(propsUtilClass, "_configuration");
      this.configurationsField = ReflectionUtil.getDeclaredField(propsUtilClass, "_configurations");
      Constructor<?> c = ReflectionUtil.getDeclaredConstructor(ConfigurationImpl.class, ClassLoader.class, String.class);
      if (c == null) {
        // since liferay-ce-portal-7.3.0-ga1
        c = ReflectionUtil.getDeclaredConstructor(ConfigurationImpl.class, ClassLoader.class, String.class, long.class, String.class);
      }
      this.configurationConstructor = c;
      if (configurationConstructor == null) {
        log.error("Failed to find suitable constructor in ConfigurationImpl, reloading of properties won't work.");
      }
    }
    
    public void jrClearCompanyProperties() {
      Object configurations = getValueSilently(configurationsField);
      if (configurations instanceof Map) {
        ((Map<?, ?>) configurations).clear();
      }
    }

    public void jrClearProperties() {
      if (configurationConstructor == null)
        return;

      Object old = getValueSilently(configurationField);
      Object newConf = jrNewConfigurationImpl(propsUtilClass.getClassLoader(), PropsFiles.PORTAL);
      if (newConf == null)
        return;

      setValueSilently(configurationField, newConf);
      log.info("Update gloabl properties conf: {} -> {}", old, newConf);
    }

    public void jrAddProperties(Properties properties, Object company) {
      Configuration conf = null;

      if (company == null) {
        conf = (Configuration) ReflectionUtil.invokeStaticByArgs(propsUtilClass, "_getConfiguration");
      }
      else {
        conf = (Configuration) ReflectionUtil.invokeStaticByArgs(propsUtilClass, "_getConfiguration", company);
      }
      log.info("Properties conf: {} will have " + properties.size() + " props", conf);
      if (conf != null) {
        conf.addProperties(properties);
      }
    }
    
    private Object getValueSilently(Field field) {
      try {
        return field.get(null);
      }
      catch (Exception e) {
        log.error(e);
        return null;
      }
    }
    
    private void setValueSilently(Field field, Object obj) {
      try {
        Field modifiersField = Field.class.getDeclaredField("modifiers");
        modifiersField.setAccessible(true);
        modifiersField.setInt(field, field.getModifiers() & ~Modifier.FINAL);
        field.set(null, obj);
      }
      catch (Exception e) {
        log.error(e);
      }
    }

    @Override
    public ConfigurationImpl jrNewConfigurationImpl(ClassLoader cl, String name) {
      if (configurationConstructor == null)
        return null;

      try {
        return (ConfigurationImpl) (configurationConstructor.getParameterTypes().length == 2 ? configurationConstructor.newInstance(cl, name) :
          configurationConstructor.newInstance(cl, name, 0L, null));
      }
      catch (Exception e) {
        log.error("Failed to construct new instance of ConfigurationImpl." , e);
      }
      return null;
    }
  }
  
}
