/**
 * Copyright (C) 2011-2012 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay.cbp;

import java.util.Properties;

import org.zeroturnaround.bundled.javassist.CannotCompileException;
import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.CtMethod;
import org.zeroturnaround.bundled.javassist.NotFoundException;
import org.zeroturnaround.bundled.javassist.expr.ExprEditor;
import org.zeroturnaround.bundled.javassist.expr.MethodCall;
import org.zeroturnaround.bundled.javassist.expr.NewExpr;
import org.zeroturnaround.javarebel.ConfigurationFactory;
import org.zeroturnaround.javarebel.integration.support.CacheAwareJavassistClassBytecodeProcessor;
import org.zeroturnaround.javarebel.integration.util.JavassistUtil;
import org.zeroturnaround.jrebel.liferay.LiferayReloader;
import org.zeroturnaround.jrebel.liferay.util.PropertiesWrapper;

/**
 * Processes <code>com.liferay.portal.deploy.hot.HookHotDeployListener</code>.
 *
 * Allows hot deploy on jsp files and static resources added by a liferay hook
 * 
 * @author Andres Luuk
 */
public class HookHotDeployListenerCBP extends CacheAwareJavassistClassBytecodeProcessor {
  private static final String[] INIT_PORTAL_PROPERTIES_PARAMS = {"java.lang.String", "java.lang.ClassLoader", "com.liferay.portal.kernel.xml.Element"};

  private static final boolean TEST_MODE = ConfigurationFactory.getInstance().getBoolean("rebel.enable_test_hacks");
  
  public void process(final ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("java.util");
    cp.importPackage("com.liferay.portal.kernel.servlet");
    cp.importPackage("com.liferay.portal.kernel.upgrade.util.UpgradeProcessUtil");
    cp.importPackage("com.liferay.portal.kernel.util");
    cp.importPackage("com.liferay.portal.util");
    cp.importPackage("org.zeroturnaround.javarebel");
    cp.importPackage("org.zeroturnaround.javarebel.integration.util");
    cp.importPackage("org.zeroturnaround.jrebel.liferay.util");

    ExprEditor registerAdditionalWebResources = new ExprEditor() {
      public void edit(MethodCall m) throws CannotCompileException {
        if ("getCustomJsps".equals(m.getMethodName())) {
          m.replace("" +
              "{" +
              "  LoggerFactory.getLogger(\"Liferay\").info(\"Trying to replace getCustomJsps call\"); " +
              "  if (!MonitorUtil.isActive(" + LiferayReloader.class.getName() + ".MONITOR + servletContextName) || !_customJspBagsMap.containsKey(servletContextName)) {" +
              "    RebelServletContext servletContextPortal = (RebelServletContext)ServletContextPool.get(PortalUtil." + getContextMethodName(cp) + "());" +
              "    servletContextPortal = LiferayHookUtil.unwrap(servletContextPortal);" +
              "    LoggerFactory.getLogger(\"Liferay\").info(\"Register additional web resoruce: {} {} {}\", new Object[] {servletContextPortal, (RebelServletContext)servletContext, $3}); " +
              "    LiferayHookUtil.registerHook(servletContextPortal, (RebelServletContext)servletContext, $3);" +
              "    $proceed($$);" +
              "  }" +
              "}");
        }
      }
      @Override
      public void edit(NewExpr e) throws CannotCompileException {
        // liferay 7
        if (e.getClassName().endsWith("CustomJspBagImpl")) {
          e.replace("" +
              "{" +
              "  $_ = $proceed($$);" +
              "  LoggerFactory.getLogger(\"Liferay\").info(\"Trying to replace getCustomJsps call\"); " +
              "  if (!MonitorUtil.isActive(" + LiferayReloader.class.getName() + ".MONITOR + servletContextName)) {" +
              "    RebelServletContext servletContextPortal = (RebelServletContext)ServletContextPool.get(PortalUtil." + getContextMethodName(cp) + "());" +
              "    servletContextPortal = LiferayHookUtil.unwrap(servletContextPortal);" +
              "    LoggerFactory.getLogger(\"Liferay\").info(\"Register additional web resoruce: {} {} {}\", new Object[] {servletContextPortal, (RebelServletContext)servletContext, $2}); " +
              "    LiferayHookUtil.registerHook(servletContextPortal, (RebelServletContext)servletContext, $2);" +
              "  }" +
              "}");
        }
      }
    };

    ctClass.getDeclaredMethod("initCustomJspDir").instrument(registerAdditionalWebResources);

    CtMethod m = ctClass.getDeclaredMethod("doInvokeDeploy");
    m.instrument(registerAdditionalWebResources);

    CtMethod m2 = ctClass.getDeclaredMethod("doInvokeUndeploy");
    m2.instrument(new ExprEditor() {
      boolean notFound = true;

      public void edit(MethodCall m) throws CannotCompileException {
        if ("destroyCustomJspBag".equals(m.getMethodName())) {
          notFound = false;
          m.replace("" +
              "{" +
              "  if (!MonitorUtil.isActive(" + LiferayReloader.class.getName() + ".MONITOR + servletContextName)) {" +
              "    RebelServletContext servletContextPortal = (RebelServletContext)ServletContextPool.get(PortalUtil." + getContextMethodName(cp) + "());" +
              "    servletContextPortal = LiferayHookUtil.unwrap(servletContextPortal);" +
              "    LiferayHookUtil.unregisterHook(servletContextPortal, (RebelServletContext)servletContext);" +
              "    $proceed($$);" +
              "  }" +
              "}");
        }
        else if (notFound && "values".equals(m.getMethodName())) {
          m.replace("" +
              "{" +
              "  if ($0.containsKey(\"customJsp\") && !MonitorUtil.isActive(" + LiferayReloader.class.getName() + ".MONITOR + servletContextName)) {" +
              "    RebelServletContext servletContextPortal = (RebelServletContext)ServletContextPool.get(PortalUtil." + getContextMethodName(cp) + "());" +
              "    servletContextPortal = LiferayHookUtil.unwrap(servletContextPortal);" +
              "    LiferayHookUtil.unregisterHook(servletContextPortal, (RebelServletContext)servletContext);" +
              "  }" +
              "  $_ = $proceed($$);" +
              "}");
        }
      }
    });

    ExprEditor addTrim = new ExprEditor() {
      public void edit(MethodCall m) throws CannotCompileException {
        if ("elementText".equals(m.getMethodName()) || "getText".equals(m.getMethodName())) {
          m.replace("$_ = $0." + m.getMethodName() + "Trim($$);");
        }
      }
    };
    m.instrument(addTrim);
    
    trackPortalProperties(cp, ctClass);
    
    if (TEST_MODE) {
      forceUpgradeProcess(cp, ctClass);
    }
  }

  private CtClass getPortalUtilClass(ClassPool cp) throws NotFoundException {
    try {
      return cp.get("com.liferay.portal.kernel.util.PortalUtil");
    }
    catch (NotFoundException e) {
      return cp.get("com.liferay.portal.util.PortalUtil");
    }
  }

  private final String getContextMethodName(ClassPool cp) {
    try {
      String servletContextName = "getServletContextName";
      getPortalUtilClass(cp).getDeclaredMethod(servletContextName);
      return servletContextName;
    }
    catch (NotFoundException e) {
      return "getPathContext";
    }
  }
  

  private void trackPortalProperties(ClassPool cp, CtClass ctClass) throws Exception {
    CtMethod initMethod = ctClass.getDeclaredMethod("initPortalProperties", cp.get(INIT_PORTAL_PROPERTIES_PARAMS));

    initMethod.addLocalVariable("_jrConfName", cp.get("java.lang.String"));
    initMethod.addLocalVariable("_jrConfCl", cp.get("java.lang.ClassLoader"));
    initMethod.insertBefore("{" +
        "  _jrConfCl = null;" +
        "  _jrConfName = null;" +
        "}");
    initMethod.instrument(new ExprEditor() {
      @Override
      public void edit(MethodCall m) throws CannotCompileException {
        if ("getConfiguration".equals(m.getMethodName())) {
          m.replace("{" +
              "  $_ = $proceed($$);" +
              "  _jrConfCl = $1;" +
              "  _jrConfName = $2;" +
              "}");
        }
        else if ("getProperties".equals(m.getMethodName())) {
          m.replace(
              "if (_jrConfCl != null && _jrConfName != null) {" +
              "  $_ = new " + PropertiesWrapper.class.getName() + "($proceed($$), _jrConfCl, _jrConfName); " +
              "} else {" +
              "  $_ = $proceed($$);" +
              "}");
        }
      }
    });
  }

  private void forceUpgradeProcess(ClassPool cp, CtClass ctClass) throws CannotCompileException {
    // In Liferay 7.4.3.96-ga96, the support of upgrade.processes property in portal.properties and invocation of custom upgrade processes on deployment was dropped.
    // See: "Liferay Remove the ability to run upgrade process from Hook" https://liferay.atlassian.net/browse/LPS-195710.
    // We have some servlet tests that include an UpgradeProcess that fulfills test prerequisites like adding a portlet to portal layout.
    // The original method executing upgrade processes ReleaseLocalServiceUtil.updateRelease(servletContextName, upgradeProcesses, unfilteredPortalProperties)
    // has been also removed. UpgradeProcessUtil.upgradeProcess(buildNumber, upgradeProcesses) does not perform all the steps of the original method,
    // but it is good enough for testing purposes.
    CtMethod initPortalProperties = JavassistUtil.getDeclaredMethod(ctClass, "initPortalProperties", String.class.getName(), ClassLoader.class.getName(),
        Properties.class.getName(), Properties.class.getName());

    final boolean callsInitUpgradeProcesses[] = { false };
    initPortalProperties.instrument(new ExprEditor() {
      @Override
      public void edit(MethodCall m) throws CannotCompileException {
        if ("initUpgradeProcesses".equals(m.getMethodName())) { // UpgradeProcessUtil.initUpgradeProcesses(portletClassLoader, upgradeProcessClassNames);
          callsInitUpgradeProcesses[0] = true;
        }
      }
    });

    if (!callsInitUpgradeProcesses[0]) {
      initPortalProperties.insertAfter("" +
          "{" +
          "  if ($4.containsKey(\"release.info.build.number\") || $4.containsKey(\"upgrade.processes\")) {" +
          "    String[] upgradeProcessClassNames = StringUtil.split($4.getProperty(\"upgrade.processes\"));" +
          "    LoggerFactory.getLogger(\"Liferay\").trace(\"Upgrade process class names: {}\", Arrays.toString(upgradeProcessClassNames));" +
          "    List upgradeProcesses = UpgradeProcessUtil.initUpgradeProcesses($2, upgradeProcessClassNames);" +
          "    UpgradeProcessUtil.upgradeProcess(GetterUtil.getInteger($4.getProperty(\"release.info.build.number\")), upgradeProcesses);" +
          "  }" +
          "}");
    }
  }

  protected CachingPolicy getCachingPolicy(ClassPool cp, ClassLoader cl) {
    return CachingPolicy.keyOnFlags(JavassistUtil.hasClass(cp, "com.liferay.portal.kernel.util.PortalUtil"), TEST_MODE);
  }
}