/**
 * Copyright (C) 2012 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay.cbp;

import org.zeroturnaround.bundled.javassist.CannotCompileException;
import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.CtConstructor;
import org.zeroturnaround.bundled.javassist.CtField;
import org.zeroturnaround.bundled.javassist.CtMethod;
import org.zeroturnaround.bundled.javassist.CtNewMethod;
import org.zeroturnaround.bundled.javassist.expr.ExprEditor;
import org.zeroturnaround.bundled.javassist.expr.MethodCall;
import org.zeroturnaround.javarebel.integration.support.CacheAwareJavassistClassBytecodeProcessor;
import org.zeroturnaround.javarebel.integration.util.JavassistUtil;
import org.zeroturnaround.jrebel.liferay.PropsUtilReloader;
import org.zeroturnaround.jrebel.liferay.util.JrPropsUtil;

/**
 * Intruments <code>com.liferay.portal.util.PropsUtil</code>
 * 
 * Reloading properties in portal*.properties files
 * 
 * @author Andres Luuk
 */
public class PropsUtilCBP extends CacheAwareJavassistClassBytecodeProcessor {

  public void process(ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("java.util");
    cp.importPackage("java.net");
    cp.importPackage("com.germinus.easyconf");
    cp.importPackage("org.zeroturnaround.javarebel");
    cp.importPackage("org.zeroturnaround.javarebel.integration.util");
    cp.importPackage("org.zeroturnaround.javarebel.integration.monitor");
    cp.importPackage("com.liferay.portal.util");
    cp.importPackage("com.liferay.portal.configuration");
    cp.importPackage("com.liferay.portal.kernel.service");
    cp.importPackage("com.liferay.portal.kernel.model");
    cp.importPackage("com.liferay.portal.kernel.security.auth");
    cp.importPackage("org.zeroturnaround.jrebel.liferay.cbp");

    if (JavassistUtil.hasDeclaredField(ctClass, "_instance")) {
      // Liferay 6 and early versions of Liferay 7
      // field "_instance" holds singleton PropsUtil and all static methods redirect call to this instance
      patchSingletonPropsUtil(cp, ctClass);
    }
    else {
      // Liferay 7
      // removed singleton, all logic were moved to static method and static initializers
      patchStaticPropsUtil(cp, ctClass);
    }
  }

  private void patchStaticPropsUtil(ClassPool cp, CtClass ctClass) throws Exception {
    ctClass.addField(CtField.make(
        "private static final " + PropsUtilReloader.class.getName() + " reloader = " + PropsUtilReloader.class.getName() + ".createStaticReloader();",
        ctClass));

    JavassistUtil.removeFinalModifer(ctClass, "_configuration");

    // Register company property sources
    for (CtMethod method : ctClass.getDeclaredMethods("_getConfiguration")) {
      registerCompanySources(method);
    }
    
    // Register default property sources
    ctClass.getClassInitializer().insertAfter("{" +
        "  List sources = ((ConfigurationImplInterface)_configuration).getLoadedSources();" +
        "  reloader.registerSource(sources);" +
        "}");

    // Register external property source
    ctClass.getDeclaredMethod("addProperties", cp.get(new String[] { "java.util.Properties" })).insertAfter("" +
        "{" +
        "  if (_configuration != _getConfiguration()) {" +
        "    try {" +
        "      long companyId = CompanyThreadLocal.getCompanyId().longValue();" +
        "      Company comp = CompanyLocalServiceUtil.getCompany(companyId);" +
        "      if (comp != null) {" +
        "        reloader.registerExternalSource($1, comp);" +
        "      } else {" +
        "        reloader.registerExternalSource($1);" +
        "      }" +
        "    } catch (Exception e) {" +
        "      LoggerFactory.getLogger(\"Liferay\").warn(\"Company lookup failed with: \" + e);" +
        "      reloader.registerExternalSource($1);" +
        "    }" +
        "  } else {" +
        "    reloader.registerExternalSource($1);" +
        "  }" +
        "} ");

    ctClass.getDeclaredMethod("addProperties", cp.get(new String[]{"com.liferay.portal.kernel.model.Company", "java.util.Properties"})).insertBefore(
          "{" + 
          "  reloader.registerExternalSource($2, $1);" +
          "} ");
  }

  private void patchSingletonPropsUtil(ClassPool cp, CtClass ctClass) throws Exception {
    // Implement JrPropsUtil Interface
    ctClass.addInterface(cp.get(JrPropsUtil.class.getName()));

    ctClass.addMethod(CtNewMethod.make(
        "public void jrClearProperties() {" +
        "  _configuration = new ConfigurationImpl(PropsUtil.class.getClassLoader(), PropsFiles.PORTAL);" +
        "}"
        , ctClass));

    final boolean hasGetConfigurationMethod = JavassistUtil.hasDeclaredMethod(cp, ctClass, "_getConfiguration", null);
    ctClass.addMethod(CtNewMethod.make(
        "public void jrAddProperties(Properties properties, Object company) {" +
            (hasGetConfigurationMethod ? "_getConfiguration()" : "_configuration") + " .addProperties(properties);" +
        "}"
        , ctClass));

    ctClass.addMethod(CtNewMethod.make(
        "public ConfigurationImpl jrNewConfigurationImpl(ClassLoader cl, String name) {" +
        "  return new ConfigurationImpl(cl, name);" +
        "}"
        , ctClass));

    boolean companyResources = JavassistUtil.hasDeclaredField(ctClass, "_configurations");
    ctClass.addMethod(CtNewMethod.make(
        "public void jrClearCompanyProperties() {" +
        (companyResources ? "if (_configurations != null) _configurations.clear();" : "") +
        "}", ctClass));

    
    // Create reloader as instance field
    ctClass.addField(CtField.make("private " + PropsUtilReloader.class.getName() + " reloader = null;", ctClass));
    CtConstructor[] cs = ctClass.getDeclaredConstructors();
    for (int i = 0; i < cs.length; i++) {
      cs[i].insertAfter("{" +
          "  reloader = new " + PropsUtilReloader.class.getName() + "(this);" +
          "  List sources = ((ConfigurationImplInterface)_configuration).getLoadedSources();" +
          "  reloader.registerSource(sources);" +
          "  reloader.start();" +
          "}");
    }
    
    // Register company property sources
    for (CtMethod method : ctClass.getDeclaredMethods("_getConfiguration")) {
      registerCompanySources(method);
    }
    
    // Register external property source
    ctClass.getDeclaredMethod("_addProperties", cp.get(new String[]{"java.util.Properties"})).insertBefore(
          "{" + 
          "  reloader.registerExternalSource($1);" +
          "} ");
  }
  
  private void registerCompanySources(CtMethod configurationMethod) throws CannotCompileException {
    configurationMethod.instrument(new ExprEditor() {
      public void edit(MethodCall m) throws CannotCompileException {
        if ("put".equals(m.getMethodName())) {
          m.replace("{" + //
              "  $_ = $proceed($$);" +
              "  List sources = ((ConfigurationImplInterface)$2).getLoadedSources();" +
              "  reloader.registerCompanySources(sources);" +
              "}");
        }
      }
    });
  }
}
