/**
 * Copyright (C) 2011-2012 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay;

import static org.zeroturnaround.javarebel.integration.util.MiscUtil.identityToString;

import javax.servlet.ServletContext;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.zeroturnaround.javarebel.ConfigurationFactory;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.RequestIntegrationFactory;
import org.zeroturnaround.javarebel.RequestListener;
import org.zeroturnaround.javarebel.StopWatch;
import org.zeroturnaround.javarebel.integration.generic.RequestListenerAdapter;
import org.zeroturnaround.javarebel.integration.monitor.MonitoredResource;
import org.zeroturnaround.javarebel.integration.util.MonitorUtil;
import org.zeroturnaround.javarebel.integration.util.ReRunThrottler;
import org.zeroturnaround.javarebel.integration.util.ReflectionUtil;
import org.zeroturnaround.javarebel.integration.util.RequestListenerUtil;
import org.zeroturnaround.javarebel.integration.util.ResourceUtil;
import com.liferay.portal.kernel.deploy.hot.HotDeployEvent;
import com.liferay.portal.kernel.deploy.hot.HotDeployException;
import com.liferay.portal.kernel.deploy.hot.HotDeployListener;
import com.liferay.portal.kernel.plugin.PluginPackage;

/**
 * Monitors Liferay resources to call re-init if something changes
 * 
 * @author Andres Luuk
 */
public class LiferayReloader {

  public static final String MONITOR = "LiferayPluginReload";

  private static final Logger log = LoggerFactory.getInstance().productPrefix(LiferayPlugin.PRODUCT_PREFIX);
  private static final boolean ENABLED = ConfigurationFactory.getInstance().getBoolean("rebel.liferay.monitor", true);

  private final HotDeployListener listener;
  private final WeakReference<ClassLoader> contextClassLoader;
  private final boolean isWebAppClassloader;
  private final WeakReference<ServletContext> servletContext;
  private final String servletContextName;
  private final PluginPackage pluginPackage;
  private final List<MonitoredResource> resources = Collections.synchronizedList(new ArrayList<MonitoredResource>());
  private final RequestListener registeredListener;

  private final RequestListenerAdapter rl = new RequestListenerAdapter(0) {

    private final ReRunThrottler throttler = new ReRunThrottler();

    @Override
    public void beforeRequest() throws Exception {
      if (throttler.shouldThrottle())
        return;

      StopWatch sw = log.createStopWatch("LiferayReloader#beforeRequest");
      try {
        LiferayReloader.this.beforeRequest();
      }
      finally {
        sw.stop();
      }
    }
  };

  public static void registerResource(Map<ServletContext, LiferayReloader> reloaders, HotDeployListener listener, HotDeployEvent event, String resource) {
    URL url = event.getContextClassLoader().getResource(resource + ".properties");
    registerResource(reloaders, listener, event, url);
  }

  public static void registerResource(Map<ServletContext, LiferayReloader> reloaders, HotDeployListener listener, HotDeployEvent event, URL url) {
    if (!ENABLED || url == null) {
      return;
    }

    ServletContext servletContext = event.getServletContext();
    try {
      LiferayReloader reloader = reloaders.get(servletContext);
      if (reloader == null) {
        reloader = new LiferayReloader(listener, event);
        reloaders.put(servletContext, reloader);
      }
      reloader.addResource(url);
    }
    catch (Exception e) {
      log.error("registerResource failed on " + listener.getClass() + " for " + identityToString(servletContext), e);
    }
  }

  private LiferayReloader(HotDeployListener listener, HotDeployEvent event) {
    this.listener = listener;
    this.contextClassLoader = new WeakReference<ClassLoader>(event.getContextClassLoader());
    this.isWebAppClassloader = isWebAppClassLoader(event.getContextClassLoader());
    this.servletContext = new WeakReference<ServletContext>(event.getServletContext());
    this.servletContextName = event.getServletContextName();
    this.pluginPackage = event.getPluginPackage();

    if (servletContextName == null)
      throw new IllegalArgumentException("servletContextName");

    registeredListener = RequestListenerUtil.bindContextClassLoader(rl);
    RequestIntegrationFactory.getInstance().addRequestListener(registeredListener);
    
    log.info("Registered reloader on {} ({}), listener {}",
        identityToString(event.getServletContext()),
        servletContextName,
        listener.getClass().getName());
  }

  void addResource(URL url) {
    resources.add(new MonitoredResource(ResourceUtil.asResource(url)));
    log.info("Listening '" + url + "' for changes on " + listener.getClass() + " in " + servletContextName);
  }

  void beforeRequest() {
    ClassLoader contextClassLoader = this.contextClassLoader.get();
    ServletContext servletContext = this.servletContext.get();

    if (servletContext == null || contextClassLoader == null) {
      RequestIntegrationFactory.getInstance().removeRequestListener(registeredListener);
      return;
    }

    if (isStopped(contextClassLoader)) {
      log.trace("Removing listener from old instance of " + servletContextName + ", cl stopped " + identityToString(contextClassLoader));
      RequestIntegrationFactory.getInstance().removeRequestListener(registeredListener);
      return;
    }

    log.trace("Checking {} resources in {} {}", resources.size(), servletContextName, listener.getClass());
    if (hasModifiedResources()) {
      StopWatch swrd = log.createStopWatch("LiferayReloader#redeploy");
      MonitorUtil.enter(MONITOR + servletContextName);
      try {
        log.info("Starting redeploy of " + servletContextName);
        redeploy(contextClassLoader, servletContext);
      }
      finally {
        MonitorUtil.exit(MONITOR + servletContextName);
        swrd.stop();
      }
    }
  }

  private boolean isStopped(ClassLoader contextClassLoader) {
    if (!isWebAppClassloader) { // only catalina WebAppClassloader can be stopped
      return false;
    }

    try {
      return !(Boolean) ReflectionUtil.invoke(contextClassLoader, "isStarted");
    }
    catch (Exception e) {
      log.trace("Failed to check isStarted() on " + identityToString(contextClassLoader), e);
      return false;
    }
  }

  private boolean isWebAppClassLoader(ClassLoader cl) {
    return "org.apache.catalina.loader.WebappClassLoader".equals(cl.getClass().getName());
  }

  private boolean hasModifiedResources() {
    boolean modified = false;
    for (MonitoredResource res : new ArrayList<MonitoredResource>(resources)) {
      try {
        if (res.modified()) {
          if (log.isTraceEnabled()) {
            log.trace("Modified: " + res);
          }
          modified = true;
        }
      } catch (Exception e) {
        if (log.isEnabled()) {
          log.log("Resource has been removed: " + res);
        }
        resources.remove(res);
      }
    }
    return modified;
  }

  private void redeploy(ClassLoader contextClassLoader, ServletContext servletContext) {
    List<MonitoredResource> resourceOld = new ArrayList<MonitoredResource>(resources);
    try {
      resources.clear();
      HotDeployEvent event = new HotDeployEvent(servletContext, contextClassLoader);
      event.setPluginPackage(pluginPackage);

      try {
        markContextForReload(servletContext.getServletContextName());
        listener.invokeUndeploy(event);
        listener.invokeDeploy(event);
      } finally {
        markContextReloaded(servletContext.getServletContextName());//even if it fails to reload, mark it as reloaded as not to block needlessly
      }
    }
    catch (IllegalStateException e) {
      RequestIntegrationFactory.getInstance().removeRequestListener(registeredListener);
      log.info("Redeploy aborted on " + listener.getClass() + " for " + identityToString(servletContext));
    }
    catch (HotDeployException e) {
      log.error("Reloading failed on " + listener.getClass() + " for " + identityToString(servletContext), e);
      resources.clear();
      resources.addAll(resourceOld);
    }
  }

  private static void markContextForReload(String contextPath) {
    try {
      Class<?> clazz = LiferayReloader.class.getClassLoader().loadClass("org.zeroturnaround.jrebel.liferay.util.LiferayDeployStatus");
      ReflectionUtil.invokeStaticByArgs(clazz, "setReloading", contextPath);
    }
    catch (ClassNotFoundException e) {
    }
  }

  private static void markContextReloaded(String contextPath) {
    try {
      Class<?> clazz = LiferayReloader.class.getClassLoader().loadClass("org.zeroturnaround.jrebel.liferay.util.LiferayDeployStatus");
      ReflectionUtil.invokeStaticByArgs(clazz, "setContextPortalEnvInitialized", contextPath);
    }
    catch (ClassNotFoundException e) {
    }
  }
}