/**
 * Copyright (C) 2011-2012 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay.cbp;

import org.zeroturnaround.bundled.javassist.CannotCompileException;
import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.CtConstructor;
import org.zeroturnaround.bundled.javassist.CtField;
import org.zeroturnaround.bundled.javassist.CtNewMethod;
import org.zeroturnaround.bundled.javassist.NotFoundException;
import org.zeroturnaround.bundled.javassist.expr.ExprEditor;
import org.zeroturnaround.bundled.javassist.expr.MethodCall;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.integration.support.CacheAwareJavassistClassBytecodeProcessor;
import org.zeroturnaround.javarebel.integration.util.JavassistUtil;
import org.zeroturnaround.jrebel.liferay.LiferayPlugin;

/**
 * Fix a bug in liferay resource closing and refreshes EasyConf
 *
 * @author Andres Luuk
 */
public class ConfigurationImplCBP extends CacheAwareJavassistClassBytecodeProcessor {

  private static final Logger log = LoggerFactory.getInstance().productPrefix(LiferayPlugin.PRODUCT_PREFIX);

  public void addLogger(CtClass ctClass) throws CannotCompileException {
    ctClass.addField(CtField.make("private static final Logger jrLogger = LoggerFactory.getInstance().productPrefix(\"" + LiferayPlugin.PRODUCT_PREFIX + "\");", ctClass));
  }

  public void process(ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("org.zeroturnaround.javarebel");

    addLogger(ctClass);
    processResources(cp, cl, ctClass);
    implementConfigurationImplInterface(cp, cl, ctClass);
  }

  public void processResources(ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("java.io");
    cp.importPackage("com.germinus.easyconf");

    CtConstructor[] cs = ctClass.getConstructors();
    for (int i = 0; i < cs.length; i++) {
      cs[i].instrument(new ExprEditor() {
        public void edit(MethodCall m) throws CannotCompileException {
          if ("load".equals(m.getMethodName())) {
            m.replace("" +
                "{" +
                "  InputStream is = (InputStream)$1;" +
                "  $_ = $proceed($$);" +
                "  try { if (is!= null) is.close(); } catch( Exception e ) {}" +
                "}");
          }
          else if ("getConfiguration".equals(m.getMethodName())) {
            try {
              if (m.getMethod().getParameterTypes().length == 1) {
                m.replace("" +
                    "{" +
                    "  StopWatch sw = jrLogger.createStopWatch(\"Liferay#refreshComponentFirst\"); " +
                    "  try {" +
                    "    EasyConf.refreshComponent($1);" +
                    "    $_ = $proceed($$);" +
                    "  } finally {" +
                    "    sw.stop();" +
                    "  }" +
                    "}");
              }
              else {
                m.replace("" +
                    "{" +
                    "  StopWatch sw = jrLogger.createStopWatch(\"Liferay#refreshComponentSecond\"); " +
                    "  try {" +
                    "    EasyConf.refreshComponent($1 + $2);" +
                    "    $_ = $proceed($$);" +
                    "  } finally {" +
                    "    sw.stop();" +
                    "  }" +
                    "}");
              }
            }
            catch (NotFoundException e) {
              if (log.isEnabled()) {
                log.error(e);
              }
            }
          }
        }
      });
    }
  }

  public void implementConfigurationImplInterface(ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("java.util");

    ctClass.addInterface(cp.get(ConfigurationImplInterface.class.getName()));
    boolean hasGetLoadedSources = JavassistUtil.hasDeclaredMethod(ctClass, "getLoadedSources");
    log.debug("ConfigurationImpl has getLoadedSources: {}", hasGetLoadedSources);
    if (hasGetLoadedSources) {
      // Looks like glassfish added their own implementation that is exactly variant 2 of what this
      // instrumentation tries to add.
      return;
    }

    if (JavassistUtil.hasDeclaredField(ctClass, "_componentProperties")) {
      ctClass.addMethod(CtNewMethod.make("" +
          "public List getLoadedSources() { " +
          "  return _componentProperties.getLoadedSources();" +
          "}", ctClass));
    }
    else if (JavassistUtil.hasDeclaredField(ctClass, "_classLoaderAggregateProperties")) {
      ctClass.addMethod(CtNewMethod.make("" +
          "public List getLoadedSources() { " +
          "  return _classLoaderAggregateProperties.loadedSources();" +
          "}", ctClass));
    }
    else {
      ctClass.addMethod(CtNewMethod.make("" +
          "public List getLoadedSources() { " +
          "  return getComponentProperties().getLoadedSources();" +
          "}", ctClass));
    }
  }
}
