/**
 * Copyright (C) 2011 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay.cbp;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.zeroturnaround.bundled.javassist.CannotCompileException;
import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.CtMethod;
import org.zeroturnaround.bundled.javassist.expr.ExprEditor;
import org.zeroturnaround.bundled.javassist.expr.MethodCall;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.integration.support.CacheAwareJavassistClassBytecodeProcessor;
import org.zeroturnaround.javarebel.integration.util.JavassistUtil;
import org.zeroturnaround.jrebel.liferay.LiferayPlugin;
import org.zeroturnaround.jrebel.liferay.LiferayReloader;
import org.zeroturnaround.jrebel.liferay.util.XMLFormatter;

import com.liferay.portal.kernel.plugin.PluginPackage;
import com.liferay.portal.kernel.xml.Element;

/**
 * JRebel portlet re-init when portlet already exists
 *
 * @author Andres Luuk
 */
public class PortletLocalServiceImplCBP extends CacheAwareJavassistClassBytecodeProcessor {

  private static final Logger log = LoggerFactory.getInstance().productPrefix(LiferayPlugin.PRODUCT_PREFIX);

  public void process(final ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("com.liferay.portlet");
    cp.importPackage("com.liferay.portal.kernel.portlet");
    cp.importPackage("org.zeroturnaround.javarebel");
    cp.importPackage("org.zeroturnaround.javarebel.integration.util");

    final CtMethod initWarMethod = ctClass.getDeclaredMethod("initWAR");
    final CtMethod initEARMethod = ctClass.getDeclaredMethod("initEAR");
    if (JavassistUtil.hasDeclaredMethod(cp, ctClass, "initWAR", new String[] { String.class.getName(), "jakarta.servlet.ServletContext", String[].class.getName(), "com.liferay.portal.kernel.plugin.PluginPackage" }) ||
        JavassistUtil.hasDeclaredMethod(cp, ctClass, "initWAR", new String[] { String.class.getName(), "javax.servlet.ServletContext", String[].class.getName(), "com.liferay.portal.kernel.plugin.PluginPackage" })) {
      formatXmlFiles(initWarMethod, "$3"); // Liferay 5.2.0+
      formatXmlFiles(initEARMethod, "$2");
    }
    else {
      formatXmlFiles(initWarMethod, "$2"); // Liferay < 5.1.2
      formatXmlFiles(initEARMethod, "$1");
    }

    initWarMethod.instrument(new ExprEditor() {
      public void edit(MethodCall m) throws CannotCompileException {
        if ("clear".equals(m.getMethodName())) {
          m.replace("" +
              "{" +
              "  if (!MonitorUtil.isActive(" + LiferayReloader.class.getName() + ".MONITOR + servletContextName)) {" + //
              "    $proceed($$);" +
              "  }" +
              "}");
        }
        else if ("destroy".equals(m.getMethodName()) && !"com.liferay.portlet.PortletConfigFactoryUtil".equals(m.getClassName())) {
          m.replace("" +
              "{" +
              "  if (!MonitorUtil.isActive(" + LiferayReloader.class.getName() + ".MONITOR + servletContextName)) {" +
              "    $proceed($$);" +
              "  }" +
              "}");
        }
      }
    });

    CtMethod readPortletXmlMethod = findReadPortletXmlMethod(cp, ctClass);
    if (readPortletXmlMethod == null) {
      log.error("Cannot instrument PortletLocalServiceImpl, method '_readPortletXML' not found!");
      return;
    }
    
    readPortletXmlMethod.instrument(new ExprEditor() {
      public void edit(MethodCall m) throws CannotCompileException {
        if ("setTimestamp".equals(m.getMethodName()) || "setPortletApp".equals(m.getMethodName())) {
          try {
            m.replace("" + //
                "{" + //
                "  portletModel.getAssetRendererFactoryClasses().clear();" + //
                "  portletModel.getCustomAttributesDisplayClasses().clear();" + //
                "  portletModel.getWorkflowHandlerClasses().clear();" + //
                "  portletModel.getHeaderPortalCss().clear();" + //
                "  portletModel.getHeaderPortletCss().clear();" + //
                "  portletModel.getHeaderPortalJavaScript().clear();" + //
                "  portletModel.getHeaderPortletJavaScript().clear();" + //
                "  portletModel.getFooterPortalCss().clear();" + //
                "  portletModel.getFooterPortletCss().clear();" + //
                "  portletModel.getFooterPortalJavaScript().clear();" + //
                "  portletModel.getFooterPortletJavaScript().clear();" + //
                "  $proceed($$);" + //
                "}");
          }
          catch (CannotCompileException e) {
            // Liferay 5
            m.replace("" + //
                "{" + //
                "  portletModel.getHeaderPortalCss().clear();" + //
                "  portletModel.getHeaderPortletCss().clear();" + //
                "  portletModel.getHeaderPortalJavaScript().clear();" + //
                "  portletModel.getHeaderPortletJavaScript().clear();" + //
                "  portletModel.getFooterPortalCss().clear();" + //
                "  portletModel.getFooterPortletCss().clear();" + //
                "  portletModel.getFooterPortalJavaScript().clear();" + //
                "  portletModel.getFooterPortletJavaScript().clear();" + //
                "  $proceed($$);" + //
                "}");
          }
        }
        else if ("setPortletClass".equals(m.getMethodName())) {
          try {
            m.replace("" + //
                "{" + //
                "  if (portletModel.getPortletClass() != null && !portletModel.getPortletClass().equals($1)) {" + //
                "    Logger log = LoggerFactory.getInstance().productPrefix(\"" + LiferayPlugin.PRODUCT_PREFIX + "\");" + //
                "    if (log.isTraceEnabled()) {" + //
                "      log.trace(\"Reinit portlet class for '\" + servletContextName + \" old=\" + portletModel.getPortletClass() + \" new=\" + $1);" + //
                "    }" + //
                "    PortletInstanceFactoryUtil.clear(portletModel);" + //
                "  }" + //
                "  $proceed($$);" + //
                "}");
          }
          catch (CannotCompileException e) {
            // Older Liferay, no PortletInstanceFactoryUtil jet
            if (log.isTraceEnabled()) {
              log.trace("setPortletClass: " + e.getMessage());
            }
          }
        }
        else if ("put".equals(m.getMethodName())) {
          try {
            m.replace("" + //
                "{" + //
                "  $_ = $proceed($$);" + //
                "  if ($2 != null && $2 instanceof String && !$2.equals($_)){" + //
                "    Logger log = LoggerFactory.getInstance().productPrefix(\"" + LiferayPlugin.PRODUCT_PREFIX + "\");" + //
                "    if (log.isTraceEnabled()) {" + //
                "      log.trace(\"Reinit portlet param for '\" + servletContextName + \" old=\" + $_ + \" new=\" + $2);" + //
                "    }" + //
                "    PortletInstanceFactoryUtil.clear(portletModel);" + //
                "  }" + //
                "}");
          }
          catch (CannotCompileException e) {
            // Older Liferay, to many maches
            if (log.isTraceEnabled()) {
              log.trace("put: " + e.getMessage());
            }
          }
        }

      }
    });
  }
  
  private static void formatXmlFiles(CtMethod initMethod, String xmlArray) throws CannotCompileException {
    initMethod.insertBefore("" +
        "{" +
        "  String[] xmls = " + xmlArray + ";" +
        "  for (int i = 0; i < xmls.length; i++) {" +
        "    xmls[i] = " + XMLFormatter.class.getName() + ".format(xmls[i]);" +
        "  }" +
        "}");
  }

  private static CtMethod findReadPortletXmlMethod(ClassPool cp, CtClass ctClass) {
    CtMethod m = null;
    // 7.4.2 ga3
    m = JavassistUtil.getFirstExistingMethod(m, cp, ctClass, "readPortletXML", String.class.getName(), PluginPackage.class.getName(), "com.liferay.portal.kernel.model.PortletApp",
        "com.liferay.portal.kernel.xml.Element", Map.class.getName(), Set.class.getName());
    // 7.0 ga1
    m = JavassistUtil.getFirstExistingMethod(m, cp, ctClass, "readPortletXML", String.class.getName(), PluginPackage.class.getName(), "com.liferay.portal.kernel.model.PortletApp",
        "com.liferay.portal.kernel.xml.Element", Map.class.getName());
    // Current Liferay snapshot from GitHub
    m = JavassistUtil.getFirstExistingMethod(m, cp, ctClass, "_readPortletXML", String.class.getName(), Map.class.getName(), PluginPackage.class.getName(), "com.liferay.portal.model.PortletApp",
        Set.class.getName(), Element.class.getName());
    // all 6.x and 7.x, not sure about exact versions -- Viktor
    m = JavassistUtil.getFirstExistingMethod(m, cp, ctClass, "_readPortletXML", String.class.getName(), Map.class.getName(), PluginPackage.class.getName(), "com.liferay.portal.model.PortletApp",
        Set.class.getName(), long.class.getName(), Element.class.getName());
    // 5.2.4
    m = JavassistUtil.getFirstExistingMethod(m, cp, ctClass, "_readPortletXML", String.class.getName(), "javax.servlet.ServletContext", String.class.getName(), Map.class.getName(), List.class.getName(), PluginPackage.class.getName());
    // 5.1.2
    m = JavassistUtil.getFirstExistingMethod(m, cp, ctClass, "_readPortletXML", String.class, String.class, Map.class, List.class, PluginPackage.class);
    return m;
  }
}