/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

import java.util.Collection;

/**
 * A callback that will be notified when a class event occurs. 
 * 
 * @since 1.2
 * 
 * @author Jevgeni Kabanov (ekabanov@zeroturnaround.com)
 */
public interface ClassEventListener {

  public enum ChangeType {
    HIERARCHY, HIERARCHY_SUPERCLASS, HIERARCHY_INTERFACE, // superclass or interface changed
    FIELD, FIELD_STATIC, FIELD_INSTANCE, // added, removed or changed field type
    METHOD, METHOD_STATIC, METHOD_INSTANCE, METHOD_CONSTRUCTOR, // added, removed or changed method interface
    ANNOTATION, ANNOTATION_CLASS, ANNOTATION_FIELD, ANNOTATION_METHOD, // class, field, method, method parameter annotation
    ONLY_BODY // Only method body changed, no visible changes in class structure
  };

  public static final int PRIORITY_DEFAULT = 0;
  public static final int PRIORITY_WEB = -1000000;
  public static final int PRIORITY_CORE = 1000000;

  /**
   * Event that occurs right after a class has been loaded.
   */
  public static final int EVENT_LOADED = 0;
  /**
   * Event that occurs right after a class has been reloaded.
   */
  public static final int EVENT_RELOADED = 1;
  /**
   * Event that occurs right after a dependency has been reloaded.
   */
  public static final int EVENT_DEPENDENCY_RELOADED = 2;
  
  /**
   * Called when a class event occurs.
   * 
   * @param eventType the event type
   * @param klass the class associated with the event
   * @param changeTypes the class associated with the event
   * @throws Exception on any issue
   * 
   * @see #EVENT_LOADED
   * @see #EVENT_RELOADED
   */
  default void onClassEvent(int eventType, Class<?> klass, Collection<ChangeType> changeTypes) throws Exception {
    onClassEvent(eventType, klass);
  }

  default void onClassEvent(int eventType, Class<?> klass) throws Exception {
  }

  /**
   * Returns the priority of the listener that defines the order of listener invocation.
   * First run the listener with greatest priority value. Negative values are the latest.
   *
   * @return the priority of the listener
   */
  default int priority() {
    return 0;
  }
}