/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

/**
 * Provides simple logging with {@link #echo(String)}-style methods outputting both to console
 * and to jrebel.log and {@link #log(String)}-style methods outputting only to log.
 * To use it acquire an instance from {@link LoggerFactory}
 * 
 * @since 1.2
 * 
 * @author Jevgeni Kabanov
 * @author Aleksei Sosnovski
 * @see LoggerFactory
 * @see Integration
 * @see Configuration
 * @see Reloader
 * @see Plugin
 */
public interface Logger {

  /**
   * Writes message to jrebel.log if enabled.
   *
   * @param message main message
   */
  default void error(String message) {
    error(message, (Object[]) null);
  }

  /**
   * Dumps stacktrace to jrebel.log if enabled.
   *
   * @param e a Throwable to dump
   */
  default void error(Throwable e) {
    error("", e);
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg a parameter to format
   */
  default void error(String format, Object arg) {
    error(format, new Object[] { arg });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg1 a parameter to format
   * @param arg2 a parameter to format
   */
  default void error(String format, Object arg1, Object arg2) {
    error(format, new Object[] { arg1, arg2 });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param args parameters to format
   */
  default void error(String format, Object... args) {
    errorEcho(format, args);
  }

  /**
   * Dumps message and stacktrace to jrebel.log if enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void error(String message, Throwable e) {
    error(message, new Object[] { e });
  }

  /**
   * Writes message to jrebel.log if enabled.
   *
   * @param message main message
   */
  default void warn(String message) {
    warn(message, (Object[]) null);
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg a parameter to format
   */
  default void warn(String format, Object arg) {
    warn(format, new Object[] { arg });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg1 a parameter to format
   * @param arg2 a parameter to format
   */
  default void warn(String format, Object arg1, Object arg2) {
    warn(format, new Object[] { arg1, arg2 });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param args parameters to format
   */
  default void warn(String format, Object... args) {
    warnEcho(format, args);
  }

  /**
   * Dumps message and stacktrace to jrebel.log if enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void warn(String message, Throwable e) {
    warn(message, new Object[] { e });
  }

  /**
   * Writes message to jrebel.log if enabled.
   *
   * @param message main message
   */
  default void info(String message) {
    info(message, (Object[]) null);
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg a parameter to format
   */
  default void info(String format, Object arg) {
    info(format, new Object[] { arg });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg1 a parameter to format
   * @param arg2 a parameter to format
   */
  default void info(String format, Object arg1, Object arg2) {
    info(format, new Object[] { arg1, arg2 });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param args parameters to format
   */
  default void info(String format, Object... args) {
    infoEcho(format, args);
  }

  /**
   * Dumps message and stacktrace to jrebel.log if enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void info(String message, Throwable e) {
    info(message, new Object[] { e });
  }

  /**
   * Writes message to jrebel.log if enabled.
   *
   * @param message main message
   */
  default void debug(String message) {
    debug(message, (Object[]) null);
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg a parameter to format
   */
  default void debug(String format, Object arg) {
    debug(format, new Object[] { arg });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg1 a parameter to format
   * @param arg2 a parameter to format
   */
  default void debug(String format, Object arg1, Object arg2) {
    debug(format, new Object[] { arg1, arg2 });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param args parameters to format
   */
  void debug(String format, Object... args);

  /**
   * Dumps message and stacktrace to jrebel.log if enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void debug(String message, Throwable e) {
    debug(message, new Object[] { e });
  }

  /**
   * Logs a message to jrebel.log if trace is enabled.
   *
   * @param message main message
   */
  default void trace(String message) {
    trace(message, (Object[]) null);
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg a parameter to format
   */
  default void trace(String format, Object arg) {
    trace(format, new Object[] { arg });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param arg1 a parameter to format
   * @param arg2 a parameter to format
   */
  default void trace(String format, Object arg1, Object arg2) {
    trace(format, new Object[] { arg1, arg2 });
  }

  /**
   * Writes formatted message to jrebel.log if enabled. Uses same format as slf4j.
   *
   * @param format main message
   * @param args parameters to format
   */
  void trace(String format, Object... args);

  /**
   * Dumps message and stacktrace to jrebel.log if enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void trace(String message, Throwable e) {
    trace(message, new Object[] { e });
  }

  /**
   * @return <code>true</code> if error logging to jrebel.log is enabled.
   */
  default boolean isErrorEnabled() {
    return true;
  }

  /**
   * @return <code>true</code> if warn logging to jrebel.log is enabled.
   */
  default boolean isWarnEnabled() {
    return true;
  }

  /**
   * @return <code>true</code> if info logging to jrebel.log is enabled.
   */
  default boolean isInfoEnabled() {
    return true;
  }

  /**
   * @return <code>true</code> if debug logging to jrebel.log is enabled.
   */
  default boolean isDebugEnabled() {
    return true;
  }

  /**
   * @return <code>true</code> if trace logging to jrebel.log is enabled.
   */
  default boolean isTraceEnabled() {
    return true;
  }

  /**
   * Writes message to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   */
  default void errorEcho(String message) {
    errorEcho(message, (Object[]) null);
  }

  void errorEcho(String message, Object... args);

  /**
   * Dumps message and stacktrace to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void errorEcho(String message, Throwable e) {
    errorEcho(message, new Object[] { e });
  }

  /**
   * Writes message to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   */
  default void warnEcho(String message) {
    warnEcho(message, (Object[]) null);
  }

  void warnEcho(String message, Object... args);

  /**
   * Dumps message and stacktrace to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void warnEcho(String message, Throwable e) {
    warnEcho(message, new Object[] { e });
  }

  /**
   * Writes message to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   */
  default void infoEcho(String message) {
    infoEcho(message, (Object[]) null);
  }

  void infoEcho(String message, Object... args);

  /**
   * Dumps message and stacktrace to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   * @param e a Throwable to format
   */
  default void infoEcho(String message, Throwable e) {
    infoEcho(message, new Object[] { e });
  }

  /**
   * Returns a StopWatch used to log time taken by something.
   *
   * @param category the name of the StopWatch
   * @return the StopWatch created
   */
  StopWatch createStopWatch(String category);

  /**
   * Returns a Logger instance including the given prefix in each message.
   * <p>
   * E.g. <code>logger.prefix("Prefix: ").log("Message")</code> produces
   * <code>Prefix: Message</code>.
   *
   * @param prefix the name for the logger
   * @return a named logger
   */
  default Logger prefix(String prefix) {
    return this;
  }

  /**
   * Returns a Logger instance including the product name plus the given prefix
   * in each message.
   * <p>
   * E.g. <code>logger.productPrefix("Prefix").log("Message")</code> produces
   * <code>JRebel-Prefix: Message</code>.
   *
   * @param prefix the name for the logger
   * @return a named logger
   */
  default Logger productPrefix(String prefix) {
    return this;
  }

  /**
   * Adds a newline to jrebel.log if enabled and prints it to console if console output is enabled.
   */
  default void echo() {
    infoEcho("");
  }

  /**
   * Logs a message to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param message main message
   */
  default void echo(String message) {
    infoEcho(message);
  }

  /**
   * Logs a message to jrebel.log if enabled and prints it to console if console output is enabled
   * including the product name (JRebel) as prefix.
   *
   * @param message main message
   */
  default void echoPrefix(String message) {
    infoEcho(message);
  }

  /**
   * Dumps stacktrace to jrebel.log if enabled and prints it to console if console output is enabled.
   *
   * @param e a Throwable to dump
   */
  default void errorEcho(Throwable e) {
    errorEcho("", e);
  }

  /**
   * Logs a message to jrebel.log if enabled.
   *
   * @param message main message
   */
  default void log(String message) {
    info(message);
  }

  /**
   * Returns <code>true</code> if logging to jrebel.log is enabled.
   * 
   * @return <code>true</code> if logging to jrebel.log is enabled.
   */
  default boolean isEnabled() {
    return true;
  }
}