/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

import java.lang.ref.Reference;

/**
 * <p>Provides control and notification over/on class loading and reloading.
 * To use acquire an instance from the {@link ReloaderFactory}.</p>
 *
 * <pre>
 * ReloaderFactory.getInstance()
 *     .addClassReloadListener(new ClassEventListener() {
 *   public void onClassEvent(int eventType, Class klass) {
 *     System.out.println(
 *       "Class '" + klass.getName() + "' was reloaded!");
 *     }
 *   });
 * </pre>
 *
 * @author Jevgeni Kabanov (ekabanov@zeroturnaround.com)
 * @see ReloaderFactory
 * @see Integration
 * @see Configuration
 * @see Logger
 * @see Plugin
 */
public interface Reloader {

  /**
   * Returns whether reloading is enabled generally.
   * @return Whether reloading is enabled generally.
   */
  boolean isReloadEnabled();

  /**
   * Returns whether this particular class is managed by JRebel and will be reloaded when it's updated.
   * @param klass Any class.
   * @return Whether this particular class is managed by JRebel and will be reloaded when it's updated.
   */
  boolean isReloadableClass(Class<?> klass);

  /**
   * Returns whether this particular class is currently being reloaded.
   * This is <code>true</code> while the {@link ClassEventListener}-s receive the events.
   * @param klass Class managed by JRebel.
   * @return whether this particular class is currently being reloaded.
   *
   * @since 3.0
   */
  boolean isReloadingClass(Class<?> klass);

  /**
   * <p>Causes a class to be reloaded if (and only if) the
   * underlying ".class" file has been updated. Returns
   * <b>true</b> if reloaded, <b>false</b> otherwise.</p>
   *
   * <p>Note, that as JRebel is lazy, such checks may be necessary
   * if you want to be sure that your metadata is up-to-date.
   * The {@link ClassEventListener} will only fire after JRebel
   * has reloaded the class, which by default occurs on method
   * calls to the class in question (and some reflection methods).</p>
   *
   * @param klass The class to check.
   * @return <b>true</b> if reloaded, <b>false</b> otherwise.
   *
   * @see ClassEventListener
   */
  boolean checkAndReload(Class<?> klass);

  /**
   * Ensure that classes are checked for changes even when previous
   * check was just performed. This method should be used with classes
   * that are generated by application server or some framework to make
   * sure they get reloaded immediately, otherwise if a reload check has
   * been just done next check may be delayed.
   */
  void forceCheck();

  /**
   * Notify that some class has been reloaded or that it will be reloaded.
   */
  void triggerClassReload();

  /**
   * In case change detection is performed in a background thread wait for
   * change detection and reload to complete.
   */
  void waitForReload();

  /**
   * Register a {@link ClassEventListener} to receive notifications when managed classes are loaded.
   *
   * @param listener a listener to add
   */
  void addClassLoadListener(ClassEventListener listener);

  /**
   * Deregister a {@link ClassEventListener}.
   *
   * @param listener a listener to remove
   */
  void removeClassLoadListener(ClassEventListener listener);

  /**
   * Register a {@link ClassEventListener} to receive notifications when managed classes are reloaded.
   *
   * @param listener a listener to add
   */
  void addClassReloadListener(ClassEventListener listener);

  /**
   * Register a {@link ClassEventListener} to receive notifications when managed classes are reloaded.
   * This method accepts Weak/Soft references.
   *
   * @param listener a reference to the listener
   */
  void addClassReloadListener(Reference<? extends ClassEventListener> listener);

  /**
   * Register a {@link ClassEventListener} to receive notifications when the given managed class is reloaded.
   *
   * @param klass the class to be associated with the listener
   * @param listener a listener to add
   */
  void addClassReloadListener(Class<?> klass, ClassEventListener listener);

  /**
   * Register a {@link ClassEventListener} to receive notifications when one of the given managed classes is reloaded.
   *
   * @param classes the classes to be associated with the listener
   * @param listener a listener to add
   */
  void addClassReloadListener(Class<?>[] classes, ClassEventListener listener);

  /**
   * Register a {@link ClassEventListener} to receive notifications when
   * one of the given managed dependent classes is reloaded.
   *
   * When the given class is being checked the dependencies are automatically checked,
   * The registered listener will be provided with the registered class
   * (not a dependency just reloaded).
   *
   * @param klass the class to be associated with the listener
   * @param dependencies the classes that trigger the event
   * @param listener a listener to add
   */
  void addDependencyReloadListener(Class<?> klass, Class<?>[] dependencies, ClassEventListener listener);

  /**
   * Register a {@link ClassEventListener} to receive notifications when the given managed class,
   * it's super class or any of it's interfaces is reloaded.
   *
   * When the given class is being checked the dependencies are automatically checked,
   * The registered listener will be provided with the registered class
   * (not a dependency just reloaded).
   *
   * @param klass the class to be associated with the listener
   * @param listener a listener to add
   */
  void addHierarchyReloadListener(Class<?> klass, ClassEventListener listener);

  /**
   * Register a {@link Runnable} to receive notifications before
   * we start reloading all the changed classes.
   * 
   * Sometimes we need to re-create some state before class reloads,
   * because some classes need to be transformed at load time based on the state
   *
   * @param beforeReloadCheck the Runnable to run before any reloads
   * 
   * @since 2018.2.3
   */
  default void addBeforeReloadCheck(Runnable beforeReloadCheck) {
  }

  /**
   * Deregister a {@link Runnable} from receive notifications before
   * we start reloading all the changed classes.
   *
   * @param beforeReloadCheck the Runnable to run before any reloads
   * 
   * @since 2023.4.1
   */
  default void removeBeforeReloadCheck(Runnable beforeReloadCheck) {
  }

  /**
   * @param klass the class to check
   * @return true if the current class is on the reload event queue, e.g. has been reloaded in the last batc
   * 
   * @since 2020.2.1
   */
  boolean hasPendingReload(Class<?> klass);

  /**
   * Deregister a {@link ClassEventListener}.
   *
   * @param listener a listener to remove
   */
  void removeClassReloadListener(ClassEventListener listener);

  /**
   * Deregister a {@link ClassEventListener}.
   * This method accepts Weak/Soft references.
   *
   * @param listener a listener reference to remove
   */
  void removeClassReloadListener(Reference<? extends ClassEventListener> listener);

  void reinitClass(Class<?> klass);

  void reinitOnReload(Class<?> klass);

  void reinitOnReload(String className);
  
  /**
   * Block reloads with priority lower than given from executing before current reload listener has completed.
   * 
   * @param priority the priority
   */
  void blockReloadListeners(int priority);

}
