/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.support;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import org.zeroturnaround.javarebel.RebelServletContext;

/**
 * Utility methods for resolving resource locations to files in the
 * file system.
 */
public abstract class ResourceUtils {
  
  /** Windows UNC file prefix */
  private static final String FILE_UNC_PREFIX = "\\\\";
  
  /** URL prefix for loading from the file system: "file:" */
  private static final String FILE_URL_PREFIX = "file:";
  
  /** URL protocol for a file in the file system: "file" */
  private static final String URL_PROTOCOL_FILE = "file";

  /** URL protocol for a file in the JBoss 5 virtual file system: "vfsfile" */
  private static final String URL_PROTOCOL_VFSFILE = "vfsfile";
  
  /** URL protocol for a file in the JBoss 6 virtual file system: "vfs" */
  private static final String URL_PROTOCOL_VFS = "vfs";

  /** URL protocol for an entry from a JBoss jar file: "vfszip" */
  public static final String URL_PROTOCOL_VFSZIP = "vfszip";
  
  /** URL protocol for an entry from a jar file: "jar" */
  private static final String URL_PROTOCOL_JAR = "jar";

  /** URL protocol for an entry from a zip file: "zip" */
  private static final String URL_PROTOCOL_ZIP = "zip";
  
  /** URL protocol for a file in the file system: "jndi" */
  private static final String URL_PROTOCOL_JNDI = "jndi";

  /** URL protocol for an entry from a WebSphere jar file: "wsjar" */
  private static final String URL_PROTOCOL_WSJAR = "wsjar";

  /** URL protocol for an entry from an OC4J jar file: "code-source" */
  private static final String URL_PROTOCOL_CODE_SOURCE = "code-source";
  
  /** Separator between JAR URL and file path within the JAR */
  private static final String JAR_URL_SEPARATOR = "!/";

  /** New separator between JAR URL and nested file path withing the JAR */
  private static final String JAR_URL_SEPARATOR_NEW = "/!";

  /**
   * Determines if the parameter is a URL with a protocol or not
   * 
   * @param s a String to check if it's URL
   * @return a true if the s was url
   */
  public static boolean isUrl(String s) {
    try {
      new URL(s); // NOSONAR
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  /**
   * Resolve the given resource URL to a <code>java.io.File</code>,
   * i.e. to a file in the file system.
   * 
   * @param resourceUrl the resource URL to resolve
   * @param sc a backup RebelServletContext to make some checks
   * @return a corresponding File object or null
   */
  public static File getFile(URL resourceUrl, RebelServletContext sc) {
    if (resourceUrl == null) {
      throw new IllegalArgumentException("Resource URL must not be null");
    }
    String protocol = resourceUrl.getProtocol();
    if (URL_PROTOCOL_JNDI.equals(protocol)) {
      // Some url resources in tomcat use jndi as protocol
      try {
        String realPath = sc.getRealPath(resourceUrl.getFile().replaceFirst(sc.getResource("/").getFile(), "/"));
        if (realPath == null) return null;
        return new File(realPath);
      }
      catch (MalformedURLException e) {
        // JNDI to File lookup failed
        throw new RuntimeException(e);
      }
    }
    return getFile(resourceUrl);
  }

  /**
   * Resolve the given resource URL to a <code>java.io.File</code>,
   * i.e. to a file in the file system.
   * 
   * @param resourceUrl the resource URL to resolve
   * @return a corresponding File object
   */
  public static File getFile(URL resourceUrl) {
    return getFile(resourceUrl, "URL");
  }

  public static File getFile(URI resourceUri) {
    if (resourceUri.isOpaque()) {
      // In case the URI is: "file:C:/work"
      try {
        return getFile(resourceUri.toURL(), "URL");
      }
      catch (MalformedURLException e) {
        throw new IllegalArgumentException("Getting file from URI '" + resourceUri + "' failed with: " + e, e);
      }
    }
    else {
      return new File(resourceUri);
    }
  }

  public static File[] getFiles(URL[] resourceUrls) {
    if (resourceUrls == null)
      throw new IllegalArgumentException("Resource URLs must not be null");
    List<File> r = new ArrayList<File>(resourceUrls.length);
    for (URL u : resourceUrls) {
      r.add(getFile(u));
    }
    return r.toArray(new File[r.size()]);
  }

  /**
   * Resolve the given resource URL to a <code>java.io.File</code>,
   * i.e. to a file in the file system.
   * @param resourceUrl the resource URL to resolve
   * @param description a description of the original resource that
   * the URL was created for (for example, a class path location)
   * @return a corresponding File object
   */
  public static File getFile(URL resourceUrl, String description) {
    if (resourceUrl == null)
      throw new IllegalArgumentException("Resource URL must not be null");

    return normalizeUNCLocalPath(new File(getSchemeSpecificPart(resourceUrl)));
  }
  
  /**
   * <p>Removes "\\localhost\" UNC part from file path.
   * 
   * <p>"\\localhost\C:\some_path" -> "C:\some_path"
   * <br>"\\127.0.0.1\C:\some_path" -> "C:\some_path"
   * 
   * @param file 
   * @return file without "\\localhost\" prefix 
   */
  private static File normalizeUNCLocalPath(File file) {
    String path = file.getPath();
    if (path.startsWith(FILE_UNC_PREFIX)) {
      int hostStartPos = FILE_UNC_PREFIX.length();
      int hostEndPos = path.indexOf('\\', hostStartPos);
      if (hostEndPos != -1) {
        String host = path.substring(hostStartPos, hostEndPos);
        if (isProbablyLocalhost(host)) {
          path = path.substring(hostEndPos);
          return new File(path);
        }
      }
    }
    return file;
  }
  
  // naive implementation, enough for our purposes
  // NB: we shouldn't do any host-resolving here
  private static boolean isProbablyLocalhost(String host) {
      return "localhost".equals(host) || host.startsWith("127.0.0.");
  }
  
  private static String getSchemeSpecificPart(URL resourceUrl) {
    try {
      return toURI(resourceUrl).getSchemeSpecificPart();
    }
    catch (URISyntaxException ex) {
      // Fallback for URLs that are not valid URIs (should hardly ever happen).
      return resourceUrl.getFile();
    }
  }
  
  /**
   * Determine whether the given URL points to a resource in the file system.
   * @param url the URL to check. May be null.
   * @return true if url references a file
   */
  public static boolean isFileURL(URL url) {
    if (url == null) {
      return false;
    }

    String protocol = url.getProtocol();
    return URL_PROTOCOL_FILE.equals(protocol)
      || (URL_PROTOCOL_VFSFILE.equals(protocol) && getFile(url).exists())
      || (URL_PROTOCOL_VFS.equals(protocol) && getFile(url).exists());
  }
  
  /**
   * Determine whether the given URL points to a resource in a jar file,
   * that is, has protocol "jar", "zip", "wsjar" or "code-source".
   * <p>"zip" and "wsjar" are used by BEA WebLogic Server and IBM WebSphere, respectively,
   * but can be treated like jar files. The same applies to "code-source" URLs on Oracle
   * OC4J, provided that the path contains a jar separator.
   * @param url the URL to check. May be null.
   * @return whether the URL has been identified as a JAR URL
   */
  public static boolean isJarURL(URL url) {
    if (url == null) {
      return false;
    }

    String protocol = url.getProtocol();
    return (URL_PROTOCOL_JAR.equals(protocol)
        || URL_PROTOCOL_ZIP.equals(protocol)
        || URL_PROTOCOL_WSJAR.equals(protocol)
        || (URL_PROTOCOL_CODE_SOURCE.equals(protocol) && url.getPath().indexOf(JAR_URL_SEPARATOR) != -1));
  }

  public static boolean isVirtualJarURL(URL url) {
    if (url == null) {
      return false;
    }
    String protocol = url.getProtocol();
    return URL_PROTOCOL_VFSZIP.equals(protocol);
  }

  /**
   * Determine if the jar entry is in a nested jar
   * @param entry Then name of the entry in the jar file
   * @return if the entry is in a nested jar
   */
  public static boolean isNestedJarEntry(final String entry) {
    return entry != null && entry.contains(JAR_URL_SEPARATOR);
  }

  /**
   * Extract the URL for the actual jar file from the given URL
   * (which may point to a resource in a jar file or to a jar file itself).
   * 
   * @param jarUrl the original URL
   * @return the URL for the actual jar file
   */
  public static URL extractJarFileURL(URL jarUrl) {
    if (jarUrl == null) {
      throw new IllegalArgumentException("JAR URL must not be null");
    }
    
    String urlFile = jarUrl.getFile();
    int separatorIndex = urlFile.indexOf(JAR_URL_SEPARATOR_NEW);
    if (separatorIndex == -1) {
      separatorIndex = urlFile.indexOf(JAR_URL_SEPARATOR);
    }
    if (separatorIndex != -1) {
      String jarFile = urlFile.substring(0, separatorIndex);
      String protocol = jarUrl.getProtocol();

      // unlike jar: urls weblogic zip: and oc4j code-source: urls don't have file: before the file name
      if (!jarFile.startsWith(FILE_URL_PREFIX) && (URL_PROTOCOL_ZIP.equals(protocol) || URL_PROTOCOL_CODE_SOURCE.equals(protocol))) {
        try {
          return new URL(FILE_URL_PREFIX + "//" + jarFile);
        } catch (MalformedURLException e) {
          throw new RuntimeException(e);
        }
      }

      try {
        return new URL(jarFile);
      }
      catch (MalformedURLException ex) {
        // Probably no protocol in original jar URL, like "jar:C:/mypath/myjar.jar".
        // This usually indicates that the jar file resides in the file system.
        if (!jarFile.startsWith("/")) {
            jarFile = "/" + jarFile;
        }
        try {
          return new URL(FILE_URL_PREFIX + jarFile);
       } catch (MalformedURLException e) {
          throw new RuntimeException(e);
        }
      }
    }
    else {
      return jarUrl;
    }
  }
  
  /**
   * Extract the JAR entry from the given URL
   * @param jarUrl the original URL
   * @return the JAR entry
   */
  public static String extractJarEntryPath(URL jarUrl) {
    if (jarUrl == null) {
      throw new IllegalArgumentException("JAR URL must not be null");
    }
    
    String urlFile = getSchemeSpecificPart(jarUrl);
    int separatorIndex = urlFile.indexOf(JAR_URL_SEPARATOR_NEW);
    if (separatorIndex != -1) {
      return urlFile.substring(separatorIndex + JAR_URL_SEPARATOR_NEW.length());
    }
    separatorIndex = urlFile.indexOf(JAR_URL_SEPARATOR);
    if (separatorIndex != -1) {
      return urlFile.substring(separatorIndex + JAR_URL_SEPARATOR.length());
    }
    return null;
  }

  /**
   * Create a URI instance for the given URL,
   * replacing spaces with "%20" quotes first.
   * <p>Furthermore, this method works on JDK 1.4 as well,
   * in contrast to the <code>URL.toURI()</code> method.
   * @param url the URL to convert into a URI instance
   * @return the URI instance
   * @throws URISyntaxException if the URL wasn't a valid URI
   * @see java.net.URL#toURI()
   */
  public static URI toURI(URL url) throws URISyntaxException {
    if (url == null) {
      throw new IllegalArgumentException("URL must not be null");
    }
    
    return toURI(url.toString());
  }

  /**
   * Create a URI instance for the given location String,
   * replacing spaces with "%20" quotes first.
   * @param location the location String to convert into a URI instance
   * @return the URI instance
   * @throws URISyntaxException if the location wasn't a valid URI
   */
  public static URI toURI(String location) throws URISyntaxException {
    return new URI(replace(location, " ", "%20"));
  }

  /**
   * Replace all occurences of a substring within a string with
   * another string.
   * @param inString String to examine
   * @param oldPattern String to replace
   * @param newPattern String to insert
   * @return a String with the replacements
   */
  private static String replace(String inString, String oldPattern, String newPattern) {
    if (inString == null) {
      return null;
    }
    if (oldPattern == null || newPattern == null) {
      return inString;
    }

    StringBuilder sbuf = new StringBuilder();
    // output StringBuffer we'll build up
    int pos = 0; // our position in the old string
    int index = inString.indexOf(oldPattern);
    // the index of an occurrence we've found, or -1
    int patLen = oldPattern.length();
    while (index >= 0) {
      sbuf.append(inString.substring(pos, index));
      sbuf.append(newPattern);
      pos = index + patLen;
      index = inString.indexOf(oldPattern, pos);
    }
    sbuf.append(inString.substring(pos));

    // remember to append any characters to the right of a match
    return sbuf.toString();
  }  
}
