/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

/**
 * A bytecode processor callback that is used by JavaRebel to process class
 * bytecode before loading the class.
 * 
 * @since 1.2
 * 
 * @author Jevgeni Kabanov
 * 
 * @see Integration  
 */
public interface ClassBytecodeProcessor {

  /**
   * @param cl the ClassLoader the class will be loaded in
   * @param classname the name of the class to be defined
   * @param bytecode original bytecode
   *
   * @return modified or unmodified bytecode.
   */
  default byte[] process(ClassLoader cl, String classname, byte[] bytecode) {
    return null;
  }

  /**
   * @param module the Module the class will be loaded in
   * @param cl the ClassLoader the class will be loaded in
   * @param classname the name of the class to be defined
   * @param bytecode original bytecode
   *
   * @return modified or unmodified bytecode.
   */
  default byte[] process(Object module, ClassLoader cl, String classname, byte[] bytecode) {
    return process(cl, classname, bytecode);
  }

  /**
   * Returns the priority of the processor that defines the order of processor invocation.
   * First run the processor with greatest priority value. Negative values are the latest.
   * 
   * @return the priority of the processor
   * 
   * @since 2018.2.5
   */
  default int priority() {
    return 0;
  }
}
