/**
 * Copyright (C) 2011 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel;

import java.io.File;
import java.util.Set;

/**
 * Abstraction of a <code>dir</code> or a <code>jar</code> element
 * defined in a <code>rebel.xml</code>.
 * 
 * @author Rein Raudjärv
 */
public interface RebelSource {
  
  /**
   * @return <code>true</code> if this is a directory source.
   */
  boolean isDir();

  /**
   * @return <code>true</code> if this is a JAR source.
   */
  boolean isJar();

  /**
   * @return the directory or JAR of this source.
   * In case of a JAR this is the JAR file.
   */
  File getFile();
  
  /**
   * @return the directory of this source. 
   * In case of a JAR this is the location where the resources are unpacked to.
   */
  File getDir();

  /**
   * @return <code>true</code> if this source uses pattern matching.
   */
  boolean hasPatterns();

  /**
   * @param path the path to check
   * 
   * @return <code>true</code> if this source may contain specified path i.e. path matches the
   * pattern used by this source.
   */
  boolean acceptPath(String path);

  /**
   * Returns all paths of this source (using pattern matching if enabled).
   * <p>
   * All paths are relative to this source.
   * Only <code>/</code> is used as a directory separator.
   * All directory paths end with a <code>/</code> character.
   * No leading characters are used.
   * 
   * @return Set&lt;String&gt; all paths of this source.
   */
  Set<String> getAllPaths();

  /**
   * Returns a listing of all direct sub-paths beginning with the given prefix.
   * <p>
   * The returned set contains both directories (ending with '/') and files.
   * 
   * @param parent parent path (prefix for all sub-paths).
   * @return direct sub-paths prefixed by the parent path or
   *   empty set if no sub-paths found (not <code>null</code>).
   */
  Set<String> getSubPaths(String parent);

}
