/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.support;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.Resource;

/**
 * A URL based resource.
 *
 * @author Jevgeni Kabanov (ekabanov@zeroturnaround.com)
 * @author Rein Raudjärv
 */
public class URLResource implements Resource {

  private final URL url;

  private boolean initialized = false;
  private File file;
  private String entry;

  public URLResource(URL url) {
    this.url = url;
  }

  public URL toURL() {
    return url;
  }

  public String toString() {
    return url.toString();
  }

  private synchronized void checkInitialized() {
    if (!initialized) {
      try {
        if (ResourceUtils.isJarURL(url)) {
          URL jarFileUrl = ResourceUtils.extractJarFileURL(url);
          if (ResourceUtils.isFileURL(jarFileUrl)) {
            file = ResourceUtils.getFile(jarFileUrl);
            entry = ResourceUtils.extractJarEntryPath(url); // NOSONAR
          }
        }
      }
      catch (Exception e) {
        file = null;
        LoggerFactory.getInstance().error(e);
      }
      initialized = true;
    }
  }

  public byte[] getBytes() {
    checkInitialized();

    JarFile jarFile = null;
    InputStream in = null;
    URLConnection con = null;

    try {
      if (file != null && !ResourceUtils.isNestedJarEntry(entry)) {
        jarFile = new JarFile(file); // NOSONAR
        JarEntry jarEntry = jarFile.getJarEntry(entry); // NOSONAR
        if (jarEntry == null)
          return null;
        in = jarFile.getInputStream(jarEntry);
      }
      else {
        con = url.openConnection();
        con.setUseCaches(false);
        con.connect();
        in = con.getInputStream();
      }
      return getBytes(in);
    }
    catch (IOException e) {
      LoggerFactory.getInstance().error(e);
    }
    finally {
      try {
        if (in != null)
          in.close();
      } catch (IOException e) {
        LoggerFactory.getInstance().error(e);
      }
      try {
        if (jarFile != null)
          jarFile.close();
      } catch (IOException e) {
        LoggerFactory.getInstance().error(e);
      }
    }

    return null;
  }

  public long lastModified() {
    checkInitialized();

    URLConnection con = null;

    try {
      if (file != null) {
        return file.lastModified();
      }
      else {
        con = url.openConnection();
        con.setUseCaches(false);
        con.connect();
        return con.getLastModified();
      }
    }
    catch (IOException e) {
      LoggerFactory.getInstance().error(e);
    }
    finally {
      try {
        if (con != null) {
          InputStream stream = con.getInputStream();
          if (stream != null) {
            stream.close();
          }
        }
      }
      catch (IOException e) {
        LoggerFactory.getInstance().error(e);
      }
      catch (NullPointerException e) { // NOSONAR
        LoggerFactory.getInstance().log("NPE when closing " + url);
        LoggerFactory.getInstance().error(e);
      }
    }

    return 0;
  }

  private static byte[] getBytes(InputStream in) throws IOException {
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    byte[] buffer = new byte[4096];
    int read;
    while ((read = in.read(buffer)) != -1) {
      out.write(buffer, 0, read);
    }
    return out.toByteArray();
  }

}
