package org.zeroturnaround.javarebel;

import java.io.IOException;
import java.io.PrintWriter;

/**
 * Http servlet response interface visible to JRebel agent.
 * <p>
 * Separate interface is required as the <code>javax.servlet.http.HttpServletResponse</code>
 * interface is not loaded by the boot class loader.
 * </p>
 * <p>
 * In practice a bytecode processor makes <code>javax.servlet.http.HttpServletResponse</code>
 * to extend the given interface.
 * </p>
 */
public interface RebelHttpServletResponse {

  /**
   * Encodes the specified URL by including the session ID in it,
   * or, if encoding is not needed, returns the URL unchanged.
   * The implementation of this method includes the logic to
   * determine whether the session ID needs to be encoded in the URL.
   * For example, if the browser supports cookies, or session
   * tracking is turned off, URL encoding is unnecessary.
   * 
   * <p>
   * For robust session tracking, all URLs emitted by a servlet should be run
   * through this method. Otherwise, URL rewriting cannot be used with browsers
   * which do not support cookies.
   *
   * @param url the url to be encoded.
   * @return the encoded URL if encoding is needed;
   *         the unchanged URL otherwise.
   */
  String encodeURL(String url);

  /**
   * Encodes the specified URL for use in the <code>sendRedirect</code> method
   * or, if encoding is not needed,
   * returns the URL unchanged. The implementation of this method
   * includes the logic to determine whether the session ID
   * needs to be encoded in the URL. Because the rules for making
   * this determination can differ from those used to decide whether to
   * encode a normal link, this method is separated from the
   * <code>encodeURL</code> method.
   * 
   * <p>
   * All URLs sent to the <code>HttpServletResponse.sendRedirect</code> method
   * should be run through this method. Otherwise, URL rewriting cannot be used
   * with browsers which do not support cookies.
   *
   * @param url the url to be encoded.
   * @return the encoded URL if encoding is needed;
   *         the unchanged URL otherwise.
   *
   * @see #sendRedirect
   * @see #encodeURL
   */
  String encodeRedirectURL(String url);

  /**
   * Sends an error response to the client using the specified
   * status. The server defaults to creating the
   * response to look like an HTML-formatted server error page
   * containing the specified message, setting the content type
   * to "text/html", leaving cookies and other headers unmodified.
   *
   * If an error-page declaration has been made for the web application
   * corresponding to the status code passed in, it will be served back in
   * preference to the suggested msg parameter.
   *
   * <p>
   * If the response has already been committed, this method throws an
   * IllegalStateException. After using this method, the response should be
   * considered to be committed and should not be written to.
   *
   * @param sc the error status code
   * @param msg the descriptive message
   * @exception IOException If an input or output exception occurs
   * @exception IllegalStateException If the response was committed
   */
  void sendError(int sc, String msg) throws IOException;

  /**
   * Sends an error response to the client using the specified status
   * code and clearing the buffer.
   * <p>
   * If the response has already been committed, this method throws an
   * IllegalStateException. After using this method, the response should be
   * considered to be committed and should not be written to.
   *
   * @param sc the error status code
   * @exception IOException If an input or output exception occurs
   * @exception IllegalStateException If the response was committed
   *              before this method call
   */
  void sendError(int sc) throws IOException;

  /**
   * Sends a temporary redirect response to the client using the
   * specified redirect location URL. This method can accept relative URLs;
   * the servlet container must convert the relative URL to an absolute URL
   * before sending the response to the client. If the location is relative
   * without a leading '/' the container interprets it as relative to
   * the current request URI. If the location is relative with a leading
   * '/' the container interprets it as relative to the servlet container root.
   *
   * <p>
   * If the response has already been committed, this method throws an
   * IllegalStateException. After using this method, the response should be
   * considered to be committed and should not be written to.
   *
   * @param location the redirect location URL
   * @exception IOException If an input or output exception occurs
   * @exception IllegalStateException If the response was committed or
   *              if a partial URL is given and cannot be converted into a valid
   *              URL
   */
  void sendRedirect(String location) throws IOException;

  /**
   * A access mirror for HttpServletResponse's method
   */
  void setStatus(int sc);

  /**
   * A access mirror for HttpServletResponse's method
   */
  void setHeader(String name, String value);

  /**
   * A access mirror for HttpServletResponse's method
   */
  PrintWriter getWriter() throws IOException;
}