/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.generic;

import org.zeroturnaround.bundled.javassist.CannotCompileException;
import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.CtConstructor;
import org.zeroturnaround.bundled.javassist.CtMethod;
import org.zeroturnaround.bundled.javassist.NotFoundException;
import org.zeroturnaround.javarebel.Integration;
import org.zeroturnaround.javarebel.integration.support.CacheAwareJavassistClassBytecodeProcessor;

/**
 * CBP for the standard class loader integration.
 *
 * <p>The given class loader is registered with JRebel at the end of any constructor.</p>
 * 
 * <p><code>findClass</code>, <code>findResource</code> and <code>findResources</code> methods
 * are delegated to JRebel.
 * {@link FindResourceClassResourceSource} is used by JRebel
 * to find resources using the given class loader. 
 * 
 * @see org.zeroturnaround.javarebel.Integration
 * @see FindResourceClassResourceSource
 * 
 * @author Jevgeni Kabanov
 * @author Rein Raudjärv
 */
public class StandardClassLoaderCBP extends CacheAwareJavassistClassBytecodeProcessor {

  public void process(ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("org.zeroturnaround.javarebel");
    cp.importPackage("org.zeroturnaround.javarebel.integration.generic");

    patchConstructors(ctClass);
    patchFindClass(cp, ctClass);
    patchFindResource(cp, ctClass);
    patchFindResources(cp, ctClass);

  }

  /**
   * Register ClassLoader with JRebel
   */
  private void patchConstructors(CtClass ctClass) throws CannotCompileException {
    CtConstructor[] constructors = ctClass.getDeclaredConstructors();
    for (int i = 0; i < constructors.length; i++) {
      if (constructors[i].callsSuper()) {
        constructors[i].insertAfter("" +
            "IntegrationFactory.getInstance().registerClassLoader($0, new FindResourceClassResourceSource($0));");
      }
    }
  }

  /**
   * Override class loading
   */
  protected void patchFindClass(ClassPool cp, CtClass ctClass) throws CannotCompileException, NotFoundException {
    try {
      CtMethod findClassMethod = ctClass.getDeclaredMethod("findClass", cp.get(new String[] { String.class.getName() }));
      findClassMethod.insertBefore("" +
          "{" +
          "  Class result = findLoadedClass($1);" +
          "  if (result != null)" +
          "    return result;" +
          "  result = IntegrationFactory.getInstance().findReloadableClass($0, $1);" +
          "  if (result != null)" +
          "   return result;" +
          "}");
    }
    catch (NotFoundException e) {
      // skip
    }
  }

  /**
   * Override finding resources
   */
  private void patchFindResource(ClassPool cp, CtClass ctClass) throws CannotCompileException {
    try {
      CtClass[] params = cp.get(new String[] { String.class.getName() });
      CtMethod method = ctClass.getDeclaredMethod("findResource", params);

      method.insertBefore("" +
          "Integration integration = IntegrationFactory.getInstance();" +
          "if (integration.isResourceReplaced($0, $1)) {" +
          "  return integration.findResource($0, $1);" +
          "}");
    }
    catch (NotFoundException e) {
      // skip
    }
  }

  private void patchFindResources(ClassPool cp, CtClass ctClass) throws CannotCompileException {
    try {
      CtClass[] params = cp.get(new String[] { String.class.getName() });
      CtMethod method = ctClass.getDeclaredMethod("findResources", params);

      method.insertBefore("" +
          "Integration integration = IntegrationFactory.getInstance();" +
          "if (integration.isResourceReplaced($0, $1)) {" +
          "  return integration.findResources($0, $1);" +
          "}");
    }
    catch (NotFoundException e) {
      // skip
    }
  }
}