/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.monitor;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.zeroturnaround.javarebel.ConfigurationFactory;
import org.zeroturnaround.javarebel.FileEventListener;
import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.RebelSource;
import org.zeroturnaround.javarebel.ResourceIntegrationFactory;
import org.zeroturnaround.javarebel.StopWatch;
import org.zeroturnaround.javarebel.integration.util.MiscUtil;
import org.zeroturnaround.javarebel.integration.util.WeakUtil;
import org.zeroturnaround.javarebel.support.ResourceUtils;

/**
 * An user of FileEventListener. Monitors the file with fsnotify if able or polls it if that feature is disabled.
 */
public class FileMonitorAdapter {
  private static final Logger log = LoggerFactory.getLogger("FileMonitorAdapter");

  private static final int CHECK_INTERVAL = ConfigurationFactory.getInstance().getCheckInterval();

  private final Set<File> monitoredFiles = Collections.synchronizedSet(new HashSet<File>());
  private final Map<File, Long> files = Collections.synchronizedMap(new HashMap<File, Long>());
  private final DirtyListener innerDirListener = new DirtyListener(false);
  private final FileEventListener listenerDir = WeakUtil.weak(innerDirListener);
  private final DirtyListener innerFileListener = new DirtyListener(false);
  private final FileEventListener listenerFile = WeakUtil.weak(innerFileListener);
  private final DirtyListener innerRecursiveListener = new DirtyListener(true);
  private final FileEventListener listenerRecursive = WeakUtil.weak(innerRecursiveListener);
  private final boolean onlyAdd;
  private volatile long lastScan = System.currentTimeMillis();
  private volatile boolean failed = false;
  private volatile boolean directories = false;

  public FileMonitorAdapter() {
    this(false);
  }

  public FileMonitorAdapter(final boolean onlyAdd) {
    this.onlyAdd = onlyAdd;
    log.info("Created " + MiscUtil.identityToString(this) + " with d=" + MiscUtil.identityToString(listenerDir) +
        " and f=" + MiscUtil.identityToString(listenerFile) + "and r=" + MiscUtil.identityToString(listenerRecursive));
  }

  /**
   * Adds a file to be monitored, those files will be checked to be up to date when dirty is called
   */
  public void addFile(File f) {
    // Normalize the file name
    try {
      f = f.getCanonicalFile();
    }
    catch (IOException e) {
      f = f.getAbsoluteFile();
    }
    if (!f.exists())
      return;
    if (!monitoredFiles.add(f))
      return;
    if (!f.isDirectory()) {
      // Remember the file time stamp to check them before final result.
      files.put(f, f.lastModified());
      if (!ResourceIntegrationFactory.getInstance().addFileListener(f.getParentFile(), listenerFile))
        failed = true;
    }
    else {
      directories = true;
      if (!ResourceIntegrationFactory.getInstance().addFileListener(f, listenerDir))
        failed = true;
    }
  }
  
  public void addRecursiveDir(URL url) {
    if (ResourceUtils.isFileURL(url)) {
      addRecursiveDir(ResourceUtils.getFile(url));
    }
  }
  
  public void addRecursiveDir(RebelSource rebelSource) {
    addRecursiveDir(rebelSource.getDir());
  }

  // In case of classpath scanning
  public void addRecursiveDir(File f) {
    directories = true;
    try {
      f = f.getCanonicalFile();
    }
    catch (IOException e) {
      f = f.getAbsoluteFile();
    }
    if (!monitoredFiles.add(f))
      return;
    if (f.isDirectory()) {
      if (!ResourceIntegrationFactory.getInstance().addFileListener(f, listenerRecursive))
        failed = true;
    }
  }

  public boolean hasAnyPaths() {
    return monitoredFiles.size() > 0;
  }

  public boolean isDirty() {
    synchronized (files) {
      if (lastScan + CHECK_INTERVAL < System.currentTimeMillis()) {
        log.trace("Checking {} for changes {}", this);
        if (failed || isAnyDirty()) {
          boolean result = failed && directories;
          if (directories && (innerDirListener.dirty || innerRecursiveListener.dirty))
            // In case we monitored directories, we can't check their time stamps because it changes only on adding and removing files (not on changing)
            result = true;
          if ((innerFileListener.dirty || failed) && !(failed && directories)) // no need to check in failed state with directories, will return dirty anyway 
            result |= checkFiles();

          reset();
          return result;
        }
        else {
          lastScan = System.currentTimeMillis();
        }
      }
    }
    return false;
  }
  
  private boolean checkFiles() {
    StopWatch sw = log.createStopWatch("DirtyCheck");
    try {
      boolean result = false;

      for (Map.Entry<File, Long> e : files.entrySet()) {
        long last = e.getKey().lastModified();
        if (last != e.getValue().longValue()) {
          e.setValue(last);
          result = true;
        }
      }
      return result;
    }
    finally {
      if (sw != null) sw.stop();
    }
  }

  private boolean isAnyDirty() {
    return innerDirListener.dirty || innerFileListener.dirty || innerRecursiveListener.dirty;
  }
  
  public void reset() {
    innerDirListener.reset();
    innerFileListener.reset();
    innerRecursiveListener.reset();
    lastScan = System.currentTimeMillis();
  }

  public String toString() {
    DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS", Locale.ENGLISH);
    return MiscUtil.identityToString(this)
        + " scanned at " + dateFormat.format(new Date(lastScan))
        + (isAnyDirty() ? " dirty" : "")
        + (failed ? " failed" : "")
        + (directories ? " directories" : "")
        + (files.size() > 0 ? " files:" + files.size() : "");
  }

  public void destroy() {
    ResourceIntegrationFactory.getInstance().removeFileListener(listenerDir);
    ResourceIntegrationFactory.getInstance().removeFileListener(listenerFile);
    ResourceIntegrationFactory.getInstance().removeFileListener(listenerRecursive);
    files.clear();
    monitoredFiles.clear();
  }

  /**
   * We must mark it dirty with every event. Because even if a file is given it might be our file but with a different path (Symbolic link fun).
   */
  private class DirtyListener implements FileEventListener {
    private final boolean recursive;
    volatile boolean dirty;

    DirtyListener(boolean recursive) {
      this.recursive = recursive;
    }

    public boolean isRecursive() {
      return recursive;
    }

    public void onFileAdd(File file) {
      if (!dirty) {
        log.info("{} detected adding of {}", MiscUtil.identityToString(this), file);
      }
      dirty = true;
    }

    public void onFileChange(File file) {
      if (!onlyAdd) {
        if (!dirty) {
          log.info("{} detected change in {}", MiscUtil.identityToString(this), file);
        }
        dirty = true;
      }
    }

    public void onFileRemove(File file) {
      if (!onlyAdd) {
        if (!dirty) {
          log.info("{} detected removing of {}", MiscUtil.identityToString(this), file);
        }
        dirty = true;
      }
    }

    public void onFileDirty(File file) {
      if (!dirty) {
        log.info("{} detected dirty event on {}", MiscUtil.identityToString(this), file);
      }
      dirty = true;
    }

    public void onFailure() {
      failed = true;
    }
    
    void reset() {
      dirty = false;
    }

    public String toString() {
      return MiscUtil.identityToString(this) + (recursive ? " is recursive" : "") + (dirty ? " is dirty" : "") + " in " + MiscUtil.identityToString(FileMonitorAdapter.this);
    }
  }
}