/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.lang.ref.WeakReference;
import java.lang.reflect.Field;
import java.util.Collection;

import org.zeroturnaround.javarebel.ClassEventListener;
import org.zeroturnaround.javarebel.ReloaderFactory;

public class ClassEventListenerUtil {

  /**
   * The target listener will be weakly referenced
   * 
   * @param listener
   * @return listener that processes events with current context class loader
   */
  public static ClassEventListener bindContextClassLoader(ClassEventListener listener) {
    return bindClassLoader(listener, getContextClassLoader());
  }

  /**
   * The target listener will be weakly referenced
   * 
   * @param listener
   * @param cl
   * @return listener that processes events with given class loader context class loader
   */
  public static ClassEventListener bindClassLoader(ClassEventListener listener, ClassLoader cl) {
    if (!canSetTCL() || cl == null)
      return WeakUtil.weakCEL(listener);
    return new BoundClassEventListener(listener, cl);
  }

  private static class BoundClassEventListener implements ClassEventListener, WeakUtil.RemovableListener {
    private final WeakReference<ClassEventListener> targetRef;
    private final WeakReference<ClassLoader> classLoderRef;
    private final String identity;
    private final int priority;

    public BoundClassEventListener(ClassEventListener target, ClassLoader cl) {
      this.targetRef = new WeakUtil.WeakListenerReference<ClassEventListener>(this, target);
      this.classLoderRef = new WeakReference<ClassLoader>(cl);
      this.priority = target.priority();
      this.identity = "boundCL(" + MiscUtil.identityToString(cl) + ")[" + MiscUtil.dumpToString(target) + "]";
    }

    private ClassLoader getClassLoader() {
      return (ClassLoader) classLoderRef.get();
    }

    private ClassEventListener getTarget() {
      return (ClassEventListener) targetRef.get();
    }

    public void onClassEvent(int eventType, Class<?> klass, Collection<ClassEventListener.ChangeType> changeTypes) throws Exception {
      ClassEventListener target = getTarget();
      if (target == null) {
        remove();
        return;
      }

      ClassLoader cl = getClassLoader();
      boolean hasCl = cl != null;
      ClassLoader old = hasCl ? setContextClassLoader(cl) : null;
      try {
        target.onClassEvent(eventType, klass, changeTypes);
      }
      finally {
        if (hasCl)
          setContextClassLoader(old);
      }
    }

    public int priority() {
      return priority;
    }

    public String toString() {
      return identity;
    }

    public void remove() {
      ReloaderFactory.getInstance().removeClassReloadListener(this);
    }
  }

  private static ClassLoader getContextClassLoader() {
    return MiscUtil.getContextClassLoader();
  }

  private static ClassLoader setContextClassLoader(ClassLoader cl) {
    if (!isOc4jThread())
      return MiscUtil.setContextClassLoader(cl);
    ClassLoader old = reflGetContextClassLoader();
    reflSetContextClassLoader(cl);
    return old;
  }

  private static final Field contextClassLoaderField = ReflectionUtil.getDeclaredField(Thread.class, "contextClassLoader");

  private static ClassLoader reflGetContextClassLoader() {
    try {
      return (ClassLoader) contextClassLoaderField.get(Thread.currentThread());
    }
    catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private static void reflSetContextClassLoader(ClassLoader cl) {
    try {
      contextClassLoaderField.set(Thread.currentThread(), cl);
    }
    catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private static boolean canSetTCL() {
    return !isOc4jThread() || contextClassLoaderField != null;
  }

  private static boolean isOc4jThread() {
    // oc4j9 server thread overrides getContextClassLoader
    return "com.evermind.server.ApplicationServerThread".equals(Thread.currentThread().getClass().getName());
  }

}
