/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.zeroturnaround.javarebel.ContainerIntegrationFactory;


/**
 * @author Rein Raudjärv
 */
public abstract class MiscUtil {
  /**
   * Return a String representation of an object's overall identity.
   * 
   * @param obj the object (may be <code>null</code>)
   * @return the object's identity as String representation,
   *         or <code>null</code> String if the object was <code>null</code>
   */
  public static String identityToString(Object obj) {
    if (obj == null)
      return "null";
    return obj.getClass().getName() + "@" + Integer.toHexString(System.identityHashCode(obj));
  }
  
  /**
   * Return a String representation of a class' overall identity.
   * @param klass the class (may be <code>null</code>)
   * @return the class' identity as String representation,
   * or <code>null</code> String if the class was <code>null</code>
   */
  public static String identityToString(Class<?> klass) {
    if (klass == null)
      return "null";
    return klass.getName() + "@" + identityToString(klass.getClassLoader());
  }
  
  /**
   * Return a String representation of a classes' overall identities.
   * @param classes the classes (may be <code>null</code>)
   * @return the classes' identities as String representation,
   * or <code>null</code> String if the class was <code>null</code>
   */
  public static String identityToString(Class<?>[] classes) {
    if (classes == null)
      return "null";
    if (classes.length == 0) {
      return "[]";
    }
    StringBuilder sb = new StringBuilder();
    sb.append("[");
    for (Class<?> klass : classes) {
      sb.append(identityToString(klass));
      sb.append(", ");
    }
    sb.setLength(sb.length() - 2);
    sb.append("]");
    return sb.toString();
  }

  public static String dumpToString(Collection<?> objects) {
    if (objects == null)
      return "null";
    if (objects.size() == 0) {
      return "[]";
    }
    StringBuilder sb = new StringBuilder();
    sb.append("[");
    for (Object o : objects.toArray()) {
      sb.append(identityToString(o));
      sb.append(", ");
    }
    sb.setLength(sb.length() - 2);
    sb.append("]");
    return sb.toString();
  }

  /**
   * Return a String representation of an object.
   * Use {@link #toString()} for classes under <code>org.zeroturnaround.javarebel</code> package.
   * Otherwise use {@link #identityToString(Object)}.
   *
   * @param obj the object (may be <code>null</code>)
   * @return a String representation of an object.
   */
  public static String dumpToString(Object obj) {
    if (obj == null)
      return "null";
    String className = obj.getClass().getName(); 
    if (className.startsWith("org.zeroturnaround.javarebel") || className.startsWith("com.zeroturnaround.javarebel"))
      return obj.toString();
    return identityToString(obj);
  }
  
  public static String toString(Collection<Object> c) {
    return c == null ? "null" : toString(c.toArray());
  }

  public static String toString(Object[] a) {
    return Arrays.toString(a);
  }

  public static String toString(Object a) {
    return String.valueOf(a);
  }

  /**
   * Adds a class to an array.
   */
  public static Class<?>[] add(Class<?>[] array, Class<?> element) {
    if (element == null)
      return array;
    if (array == null || array.length == 0)
      return new Class[] { element };
    Class<?>[] result = new Class[array.length + 1];
    System.arraycopy(array, 0, result, 0, array.length);
    result[array.length] = element;
    return result;
  }

  public static ClassLoader getSystemClassLoader() {
    return SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<ClassLoader>() {
      public ClassLoader run() {
        return ClassLoader.getSystemClassLoader();
      }
    });
  }

  public static final ClassLoader getContextClassLoader() {
    return SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<ClassLoader>() {
      public ClassLoader run() {
        return Thread.currentThread().getContextClassLoader();
      }
    });
  }

  public static final ClassLoader setContextClassLoader(final ClassLoader cl) {
    return SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<ClassLoader>() {
      public ClassLoader run() {
        Thread thread = Thread.currentThread();
        ClassLoader old = thread.getContextClassLoader();
        thread.setContextClassLoader(cl);
        return old;
      }
    });
  }

  public static boolean objectEquals(Object first, Object second) {
    return first == second || first != null && first.equals(second);
  }

  public static int indexOf(byte[] haystack, byte[] needle) {
    final byte s = needle[0];
    for (int i = 0; i <= haystack.length - needle.length; i++) {
      if (haystack[i] == s) {
        for (int j = 0; j < needle.length; j++) {
          if (haystack[i + j] != needle[j]) {
            break;
          }
          else if (j == needle.length - 1) {
            return i;
          }
        }
      }
    }
    return -1;
  }

  public static List<String> getVmArguments() {
    // Initializing the runtime MX bean will cause some early logging initialization on IBM VMs which results in wrong
    // handlers being added when compared to a run without JRebel.
    // We bypass the MX bean initialization by using an internal IBM class directly.
    List<String> lst;
    try {
      Class<?> vm = Class.forName("com.ibm.oti.vm.VM");
      Method getVMArgs = vm.getDeclaredMethod("getVMArgs");
      String[] args = (String[]) getVMArgs.invoke(null);
      lst = Arrays.asList(args);
    }
    catch (Exception ignored) {
      lst = SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<List<String>>() {
        public List<String> run() {
          RuntimeMXBean rmBean = ManagementFactory.getRuntimeMXBean();
          return rmBean.getInputArguments();
        }
      });
    }

    ArrayList<String> vmArgs = new ArrayList<String>();
    StringBuilder prev = new StringBuilder();
    for (String s : lst) {
      if (s.startsWith("-")) {
        if (prev.length() > 0) {
          vmArgs.add(prev.toString());
        }
        prev = new StringBuilder(s);
      }
      else {
        prev.append(' ').append(s);
      }
    }
    if (prev.length() > 0) {
      vmArgs.add(prev.toString());
    }
    return vmArgs;
  }

  /**
   * checks if provided file is in container temp dir, if container has temp dir integration present.
   */
  public static boolean isInContainerTempDir(final File file) {
    if (file == null) {
      return false;
    }
    Iterator<File> tempDirs = ContainerIntegrationFactory.getInstance().getContainerTempDirs().iterator();
    while (tempDirs.hasNext()) {
      File child = file;
      File tmp = (File) tempDirs.next();
      do {
        if (!tmp.equals(child)) {
          child = child.getParentFile();
        } else {
          return true;
        }
      } while (child != null);
    }
    return false;
  }

  public static int hash32(byte[] bytes) {
    return MurmurHash3.murmurhash3_x86_32(bytes, 0, bytes.length, 0x9747b28c);
  }

  public static int hash32(CharSequence data) {
    return MurmurHash3.murmurhash3_x86_32(data, 0, data.length(), 0x9747b28c);
  }
}