/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.util.HashMap;
import java.util.Map;


/**
 * Provides thread-local named counters.
 * 
 * @author Rein Raudjärv
 */
public class MonitorUtil {
  static class Counter {
    private int value = 0;
    public boolean isActive() { return value > 0; }
    public void enter() { value++; }
    public void exit() { value--; }
  }
  
  static class MonitorHolder {
    private Map<String, Counter> counters = new HashMap<String, Counter>();
    public boolean isActive(String id) {
      Counter counter = (Counter) counters.get(id);
      return counter != null && counter.isActive();
    }
    private Counter getCounter(String id) {
      return getCounter(id, false);
    }
    private Counter getCounter(String id, boolean create) {
      Counter counter = (Counter) counters.get(id);
      if (counter == null) {
        counter = new Counter();
        counters.put(id, counter);
      }
      return counter;
    }
    public void enter(String id) {
      getCounter(id, true).enter();
    }
    public void exit(String id) {
      Counter c = getCounter(id);
      if (c != null) {
        c.exit();
        if (!c.isActive())
          counters.remove(id);
      }
    }
  }
  
  private static final ThreadLocal<MonitorHolder> monitor = new ThreadLocal<MonitorHolder>() {
    protected MonitorHolder initialValue() {
      return new MonitorHolder();
    }
  };
  
  private static MonitorHolder getHolder() {
    return ((MonitorHolder) monitor.get());
  }
  
  public static boolean isActive(String id) {
    return getHolder().isActive(id);
  }
  
  public static void enter(String id) {
    getHolder().enter(id);
  }
  
  public static void exit(String id) {
    MonitorHolder holder = getHolder();
    // make sure we don't decrease counter 
    // when exit is called before enter
    if (holder.isActive(id))
      holder.exit(id);
  }

}