/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.generic;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.Collection;
import org.zeroturnaround.javarebel.ClassEventListener;
import org.zeroturnaround.javarebel.integration.util.MiscUtil;
import org.zeroturnaround.javarebel.integration.util.ReloaderUtil;
import org.zeroturnaround.javarebel.integration.util.WeakUtil;

/**
 * ClassEventListener wrapper to avoid multiple invocations for a same event.
 * <p>
 * Instead of a dependency the original class is used as the event parameter.
 * 
 * @author Rein Raudjärv
 * 
 * @see org.zeroturnaround.javarebel.integration.util.ReloaderUtil
 */
public class DependencyClassEventListener implements ClassEventListener {

  private final Reference<Class<?>> klass;
  private final Collection<Class<?>> dependencies;
  private final ClassEventListener target;

  public DependencyClassEventListener(Class<?> klass, Collection<Class<?>> dependencies, ClassEventListener target) {
    if (klass == null)
      throw new IllegalArgumentException("Class must be provided");
    if (dependencies == null || dependencies.isEmpty())
      throw new IllegalArgumentException("Dependencies must be provided");
    if (target == null)
      throw new IllegalArgumentException("target listener must be provided");
    this.klass = new WeakReference<Class<?>>(klass);
    this.dependencies = WeakUtil.createWeakHashSet(dependencies);
    this.target = target;
  }

  public void onClassEvent(int eventType, Class<?> klass, Collection<ClassEventListener.ChangeType> changeTypes) throws Exception {
    // Check other classes for being reloaded to avoid nested call
    if (ReloaderUtil.isReloadingAnyClass(dependencies, klass))
      return; // This listener is already being called

    Class<?> targetClass = this.klass.get();
    if (targetClass == null)
      return; // The target class was garbaged

    target.onClassEvent(EVENT_DEPENDENCY_RELOADED, targetClass, changeTypes);
  }

  public int priority() {
    return target.priority();
  }

  public Object getTarget() {
    return target;
  }

  public String toString() {
    return "DependencyClassEventListener[" + MiscUtil.dumpToString(target) + "]";
  }

}
