/**
 * Copyright (C) 2010 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.zeroturnaround.javarebel.integration.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;

/**
 * Helper methods for managing operations with the file system..
 * 
 * @author Rein Raudjärv
 */
public class FileUtil {
  private static final Logger log = LoggerFactory.getLogger("Util");

  /**
   * Copies the given file to a new location.
   * <p>
   * If the destination file already exists it will be overwritten.
   * 
   * @param srcFile source file (not <code>null</code>, must exist).
   * @param destFile destination file (not <code>null</code>, parent directory must exist).
   */
  public static void copyFile(File srcFile, File destFile) throws IOException {
    if (log.isEnabled())
      log.log("Copying " + srcFile + " to " + destFile);

    copyAndClose(new FileInputStream(srcFile), destFile);
  }

  /**
   * Copies the given stream to a file.
   * <p>
   * If the destination file already exists it will be overwritten.
   * 
   * @param in source stream (not <code>null</code>).
   * @param outputFile destination file (not <code>null</code>, parent directory must exist).
   */
  public static void copyAndClose(InputStream in, File outputFile) throws IOException {
    OutputStream out = new FileOutputStream(outputFile);
    try {
      byte[] buf = new byte[16384];
      int i = 0;
      while ((i = in.read(buf)) != -1)
        out.write(buf, 0, i);
    }
    finally {
      closeQuietly(in);
      closeQuietly(out);
    }
  }

  /**
   * Deletes the given file or directory.
   * 
   * Throws an IOException in case of any error.
   */
  public static void forceDelete(final File file) throws IOException {
    if (log.isEnabled())
      log.log("Deleting " + file);
    
    doForceDelete(file);
    
    if (log.isEnabled())
      log.log("Deleted " + file);
  }
  
  private static void doForceDelete(final File file) throws IOException {
    if (file.isDirectory()) {
      File[] entries = file.listFiles();
      if (entries != null)
        for (int i = 0; i < entries.length; i++)
          doForceDelete(entries[i]);
    }
    delete(file);
  }
  
  private static void delete(final File file) throws IOException {
    Boolean obj = SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<Boolean>() {
      public Boolean run() {
        return file.delete();
      }
    });
    if (!obj.booleanValue())
      throw new IOException("Could not delete " + file);
  }
  
  /**
   * Returns the extension of a given file or <code>null</code> if the file has no extension.
   * 
   * @param file input file (not <code>null</code>).
   */
  public static String getExtension(File file) {
    String path = file.getPath();
    int i = path.lastIndexOf('.');
    return i == -1 ? null : path.substring(i + 1);
  }
  
  /**
   * Close an input stream ignoring errors.
   */
  private static void closeQuietly(InputStream in) {
    if (in != null) {
      try {
        in.close();
      }
      catch (IOException e) {
      }
    }
  }
  
  /**
   * Close an output stream ignoring errors.
   */
  private static void closeQuietly(OutputStream out) {
    if (out != null) {
      try {
        out.close();
      }
      catch (IOException e) {
      }
    }
  }

  public static boolean canWriteToDir(File dir) {
    if (dir == null) return false;
    if (!(dir.exists() || dir.mkdirs())) return false;
    if (!dir.isDirectory() || !dir.canRead() || !dir.canWrite()) return false;

    String tempFileName = String.format("permissions_check_" + System.currentTimeMillis() + "_" + Math.random());
    File tempFile = new File(dir, tempFileName);
    try {
      tempFile.createNewFile(); // NOSONAR
      return tempFile.canRead() && tempFile.canWrite();
    }
    catch (IOException e) {
      return false;
    }
    catch (SecurityException e) {
      return false;
    }
    finally {
      try {
        return tempFile.delete();
      }
      catch (Exception e) {
      }
    }
  }

  public static byte[] getBytes(File f) {
    RandomAccessFile ac = null;
    try {
      // This is faster then Files.readAllBytes
      ac = new RandomAccessFile(f, "r");
      byte[] b = new byte[(int) ac.length()];
      ac.readFully(b);
      return b;
    }
    catch (FileNotFoundException e) {
      log.warn("Failed to read " + f + " with: " + e, e);
    }
    catch (IOException e) {
      log.warn("Failed to read " + f + " with: " + e, e);
    }
    finally {
      if (ac != null) {
        try {
          ac.close();
        }
        catch (IOException e) {
        }
      }
    }
    return null;
  }

  /**
   * @param files Collection of files
   * @return files containing only parent directories.
   * For example if there are [C:\weblogic\x\y, C:\weblogic\, C:\tomcat], then result is [C:\weblogic\, C:\tomcat],
   * because C:\weblogic\x\y is a subdirectory of C:\weblogic
   */
  public static Collection<File> getUniqueParentDirs(Collection<File> files) {
    if (files.size() < 2) {
      return files;
    }
    List<FileEntry> chosenFileEntries = new ArrayList<FileEntry>(files.size());
    List<FileEntry> work = new ArrayList<FileEntry>();
    List<File> result = new ArrayList<File>();

    for (File f : files) {
      try {
        work.add(new FileEntry(f));
      } catch (IOException e) {
        log.error("Unable to resolve file canonical path for " + f, e);
        // We will not exclude it, don't know if we should.
        result.add(f);
      }
    }

    Collections.sort(work);
    outerloop:
    for (FileEntry candidate : work) {
      for (FileEntry existing : chosenFileEntries) {
        if (candidate.startsWith(existing)) {
          continue outerloop;
        }
      }
      chosenFileEntries.add(candidate);
    }

    for (FileEntry existing : chosenFileEntries) {
      result.add(existing.file);
    }
    return result;
  }

  private static class FileEntry implements Comparable<FileEntry> {
    private final File file;
    private final String canonicalPath;

    private FileEntry(File file) throws IOException {
      this.file = file;
      // This fileseparator is necessary, otherwise false positives may appear with C:\xxx and C:\xxxy for example.
      this.canonicalPath = file.getCanonicalPath() + File.separator;
    }

    private boolean startsWith(FileEntry other) {
      return canonicalPath.startsWith(other.canonicalPath);
    }

    public int compareTo(FileEntry o) {
      String fname1 = canonicalPath;
      String fname2 = o.canonicalPath;
      int result = fname1.length() - fname2.length();
      if (result == 0) {
        return fname1.compareTo(fname2);
      }
      return result;
    }
  }

  public static URI toURI(final File file) {
    if (System.getSecurityManager() == null)
      return file.toURI();
    return SecurityController.doWithoutSecurityManager(new SecurityController.PrivilegedAction<URI>() {
      public URI run() {
        return file.toURI();
      }
    });
  }
}
