/**
 * Copyright (C) 2012 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.ref.WeakReference;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.zeroturnaround.javarebel.Logger;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.RebelServletContext;
import org.zeroturnaround.javarebel.RebelSource;
import org.zeroturnaround.javarebel.ResourceIntegrationFactory;
import org.zeroturnaround.javarebel.ServletIntegrationFactory;
import org.zeroturnaround.jrebel.liferay.LiferayPlugin;

public class ThemeCssReloader {

  private static final Logger log = LoggerFactory.getLogger( LiferayPlugin.PRODUCT_PREFIX);
  private static final boolean enabled = ResourceIntegrationFactory.getInstance().isEnabled();
  private static final Map<String, ThemeCssReloader> monitorsByContexts = Collections.synchronizedMap(new HashMap<String, ThemeCssReloader>());

  private final Map<String, Long> generationTimestamps = Collections.synchronizedMap(new HashMap<String, Long>());
  private final Map<String, Set<String>> cssDependencies = Collections.synchronizedMap(new HashMap<String, Set<String>>());

  private final WeakReference<RebelServletContext> context;
  private final String contextPath;
  private final List<String> roots;

  private ThemeCssReloader(RebelServletContext context, String contextPath, List<String> roots) {
    this.context = new WeakReference<RebelServletContext>(context);
    this.contextPath = contextPath;
    this.roots = roots;
  }

  public static ThemeCssReloader addToServletContext(RebelServletContext context, String contextPath) {
    ThemeCssReloader themeFilesMonitor = getInstance(contextPath);
    if (themeFilesMonitor != null) {
      if (context.equals(themeFilesMonitor.context.get())) {
        return themeFilesMonitor;
      }
      log.info("Application has been redeploy on: {}", contextPath);
    }

    themeFilesMonitor = new ThemeCssReloader(context, contextPath, findWebRoots(context));
    if (contextPath != null && contextPath.length() > 0) {
      monitorsByContexts.put(contextPath, themeFilesMonitor);
    }

    log.info("Watching roots: {} for {}", themeFilesMonitor.roots, themeFilesMonitor.contextPath);
    return themeFilesMonitor;
  }

  private static List<String> findWebRoots(RebelServletContext context) {
    List<String> roots = new ArrayList<String>();
    RebelSource[] rebelSources = ServletIntegrationFactory.getInstance().getRebelSources(context);
    if (rebelSources != null) {
      for (RebelSource rebelSource : rebelSources) {
        File dir = rebelSource.getFile();
        if (dir.isDirectory()) {
          roots.add(dir.getAbsolutePath());
        }
      }
    }

    String transparentPath = ServletIntegrationFactory.getInstance().getTransparentRealPath(context, "");
    if (transparentPath != null) {
      roots.add(transparentPath);
    }
    return roots;
  }

  public static boolean isContextMonitored(String contextPath, String servletPath) {
    return
        getInstance(contextPath) != null
            && servletPath.toLowerCase().endsWith(".css")
            && servletPath.toLowerCase().startsWith("/css/");
  }

  public static String[] remapThemePath(RebelServletContext rebelContext, Map<String, Object> inputObjects) {
    String cssThemePath = (String) inputObjects.get("cssThemePath");
    List<String> result = new ArrayList<String>();

    // map rebel sources
    for (String root : findWebRoots(rebelContext)) {
      File candidate = new File(root, "/css");
      if (candidate.exists()) {
        result.add(candidate.getAbsolutePath());
      }
    }

    // add original location as fallback
    String transparent = ServletIntegrationFactory.getInstance().getTransparentRealPath(rebelContext, "");
    String rebel = rebelContext.getRealPath("");
    if (rebel != null && transparent != null && cssThemePath.startsWith(rebel)) {
      File original = new File(transparent, cssThemePath.substring(rebel.length()));
      if (original.exists()) {
        result.add(original.getAbsolutePath());
      }
    }

    if (result.isEmpty()) {
      result.add(cssThemePath); //
    }
    else {
      log.info("Changed SASS eval path to: {}", result);
    }
    return result.toArray(new String[result.size()]);
  }

  public static void updateDependencies(String contextPath, String resourcePath, Set<String> dependencies) {
    ThemeCssReloader instance = getInstance(contextPath);
    if (instance != null) {
      instance.updatePathDependencies(resourcePath, dependencies);
    }
  }

  private void updatePathDependencies(String resourcePath, Set<String> dependencies) {
    resourcePath = sanitizePath(resourcePath);

    Set<String> relativeDependencies = new HashSet<String>();
    for (String dependency : dependencies) {
      for (String root : roots) {
        if (dependency.startsWith(root)) {
          relativeDependencies.add(dependency.substring(root.length()));
        }
      }
    }

    log.trace("Dependencies for {}{} are {}", contextPath, resourcePath, relativeDependencies);
    cssDependencies.put(resourcePath, relativeDependencies);
  }

  public static boolean isDirty(String contextPath, String requestPath, Class<?> filterClazz) {
    ThemeCssReloader instance = getInstance(contextPath);
    return instance != null && instance.isDirty(requestPath, filterClazz);
  }

  public synchronized boolean isDirty(String requestPath, Class<?> filterClazz) {
    requestPath = sanitizePath(requestPath);

    Long reloadTimestamp = generationTimestamps.get(requestPath);
    boolean dirty = reloadTimestamp == null || reloadTimestamp < getLastChangeTime(requestPath);

    String prefix = filterClazz == null ? "" : "Filter class \""+filterClazz.getSimpleName()+"\": ";
    log.debug("{}Resource {}{} is {}", prefix, contextPath, requestPath, (dirty  ? "dirty" : "clean"));

    return !enabled || dirty;
  }

  private long getLastChangeTime(String requestPath) {
    RebelServletContext context = this.context.get();
    if (context == null) {
      throw new IllegalStateException("context");
    }

    long latest = getLastModified(context, requestPath);

    Set<String> dependencies = cssDependencies.get(requestPath);
    if (dependencies != null) {
      for (String dependency : dependencies) {
        latest = Math.max(latest, getLastModified(context, dependency));
      }
    }

    return latest;
  }

  private long getLastModified(RebelServletContext context, String resourcePath) {
    InputStream closeReference = null;
    String id = null;
    try {
      URL resource = context.getResource(resourcePath);
      log.trace("Got resource {}", resource);
      URLConnection urlConnection = resource.openConnection();
      id = urlConnection.toString();
      log.trace("Opened connection to {}", id);
      long lastModified = urlConnection.getLastModified();
      closeReference = urlConnection.getInputStream();
      log.trace("Got last modification date of {}", lastModified);
      return lastModified;
    }
    catch (Exception e) {
      log.warnEcho("failed to check timestamp of '" + resourcePath + "': " + e.getMessage());
      log.error(e);
      return 0;
    }
    finally {
      if (closeReference != null) {
        try {
          closeReference.close();
          log.trace("Closed connection to {}", id);
        } catch (IOException e) {
          log.error(e);
        }
      }
    }
  }

  public static void reset(String contextPath, String resourcePath) {
    ThemeCssReloader instance = getInstance(contextPath);
    if (instance != null) {
      instance.reset(resourcePath);
    }
  }

  public synchronized void reset(String requestPath) {
    requestPath = sanitizePath(requestPath);
    generationTimestamps.put(requestPath, System.currentTimeMillis());

    log.debug("Resetting resource {}{}", contextPath, requestPath);
  }

  private static String sanitizePath(String path){
    return path.trim().replace("//", "/");
  }

  private static ThemeCssReloader getInstance(String contextPath) {
     return monitorsByContexts.get(contextPath);
  }

}
