/**
 * Copyright (C) 2011-2013 ZeroTurnaround OU
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.zeroturnaround.jrebel.liferay;

import org.zeroturnaround.javarebel.ClassResourceSource;
import org.zeroturnaround.javarebel.ConfigurationFactory;
import org.zeroturnaround.javarebel.Integration;
import org.zeroturnaround.javarebel.IntegrationFactory;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.Plugin;
import org.zeroturnaround.jrebel.liferay.cbp.AdvisedSupportAdapterCBP;
import org.zeroturnaround.jrebel.liferay.cbp.AggregateFilterCBP;
import org.zeroturnaround.jrebel.liferay.cbp.BaseAutoDeployListenerCBP;
import org.zeroturnaround.jrebel.liferay.cbp.CacheFilterCBP;
import org.zeroturnaround.jrebel.liferay.cbp.ComboServletCBP;
import org.zeroturnaround.jrebel.liferay.cbp.ConfigurationImplCBP;
import org.zeroturnaround.jrebel.liferay.cbp.CustomJspBagRegistryUtilCBP;
import org.zeroturnaround.jrebel.liferay.cbp.DelegateInvocationHandlerCBP;
import org.zeroturnaround.jrebel.liferay.cbp.DirectServletRegistryImplCBP;
import org.zeroturnaround.jrebel.liferay.cbp.DynamicCSSFilterCBP;
import org.zeroturnaround.jrebel.liferay.cbp.DynamicCSSUtilCBP;
import org.zeroturnaround.jrebel.liferay.cbp.FileUtilCBP;
import org.zeroturnaround.jrebel.liferay.cbp.FormTagCBP;
import org.zeroturnaround.jrebel.liferay.cbp.HeaderFilterCBP;
import org.zeroturnaround.jrebel.liferay.cbp.HookHotDeployListenerCBP;
import org.zeroturnaround.jrebel.liferay.cbp.HotDeployListenerCBP;
import org.zeroturnaround.jrebel.liferay.cbp.I18NCBP;
import org.zeroturnaround.jrebel.liferay.cbp.IPGeocoderImplCBP;
import org.zeroturnaround.jrebel.liferay.cbp.LiferayResourceBundleCBP;
import org.zeroturnaround.jrebel.liferay.cbp.MainServletCBP;
import org.zeroturnaround.jrebel.liferay.cbp.MinifierFilterCBP;
import org.zeroturnaround.jrebel.liferay.cbp.PortletBagFactoryCBP;
import org.zeroturnaround.jrebel.liferay.cbp.PortletContextBridgeLiferayImplCBP;
import org.zeroturnaround.jrebel.liferay.cbp.PortletContextImplCBP;
import org.zeroturnaround.jrebel.liferay.cbp.PortletHotDeployListenerCBP;
import org.zeroturnaround.jrebel.liferay.cbp.PortletLocalServiceImplCBP;
import org.zeroturnaround.jrebel.liferay.cbp.PropsUtilCBP;
import org.zeroturnaround.jrebel.liferay.cbp.RubyExecutorCBP;
import org.zeroturnaround.jrebel.liferay.cbp.ServiceBeanAopProxyCBP;
import org.zeroturnaround.jrebel.liferay.cbp.ServletContextUtilCBP;
import org.zeroturnaround.jrebel.liferay.cbp.UTF8ControlCBP;

/**
 * Supports reloading web resources from deployed hook applications to ROOT application
 * Supports realoding of xml conf files.
 *
 * Tested with Liferay 6.0.6
 *
 * @author Andres Luuk
 */
public class LiferayPlugin implements Plugin {
  public static final String PRODUCT_PREFIX = "Liferay";

  public void preinit() {
    ClassLoader cl = getClass().getClassLoader();
    Integration integration = IntegrationFactory.getInstance();

    /**
     * Adds support for reloading LiferayResourceBundle values. Language.properties file for example.
     */
    integration.addIntegrationProcessor(cl, "com.liferay.portal.language.LiferayResourceBundle", new LiferayResourceBundleCBP());

    /**
     * 1) Adds deployed Hooks rebel.xml web resources to Liferay ROOT-s path
     * 2) Registers portal properties for monitoring
     */
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.HookHotDeployListener", new HookHotDeployListenerCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.CustomJspBagRegistryUtil", new CustomJspBagRegistryUtilCBP());

    /**
     * Monitors some xml and properties files for changes
     */
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.HookHotDeployListener", new HotDeployListenerCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.ThemeLoaderHotDeployListener", new HotDeployListenerCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.ThemeHotDeployListener", new HotDeployListenerCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.LayoutTemplateHotDeployListener", new HotDeployListenerCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.PortletHotDeployListener", new HotDeployListenerCBP());

    /**
     * Reloading portlet-s xmls without portlet reinit
     */
    integration.addIntegrationProcessor(cl, "com.liferay.portal.deploy.hot.PortletHotDeployListener", new PortletHotDeployListenerCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portal.service.impl.PortletLocalServiceImpl", new PortletLocalServiceImplCBP());
    integration.addIntegrationProcessor(cl, "com.liferay.portlet.PortletBagFactory", new PortletBagFactoryCBP());

    /**
     * Fix for Liferay stream closing and refreshes EasyConf
     */
    integration.addIntegrationProcessor(cl, "com.liferay.portal.configuration.ConfigurationImpl", new ConfigurationImplCBP());
    /**
     * Fix stream closing.
     */
    integration.addIntegrationProcessor("com.liferay.portal.kernel.servlet.ServletContextUtil", new ServletContextUtilCBP().withDuplicatePatchingProtection());

    /**
     * Reloading properties in portal*.properties files
     */
    integration.addIntegrationProcessor(cl, "com.liferay.portal.util.PropsUtil", new PropsUtilCBP());

    /**
     * A Fix for a Alloy UI bug in Liferay 6.0.6
     *
     * No CL here, because this taglib is added to different portlets in jars.
     */
    integration.addIntegrationProcessor(
        "com.liferay.taglib.aui.FormTag",
        new FormTagCBP().withDuplicatePatchingProtection());

    /**
     * Fix for JRebel Spring Integration on Liferay 6.1.CE.GA1+
     */
    integration.addIntegrationProcessor(
        "com.liferay.portal.spring.aop.ServiceBeanAopProxy",
        new ServiceBeanAopProxyCBP().withDuplicatePatchingProtection());

    integration.addIntegrationProcessor(
        "com.liferay.portal.servlet.DirectServletRegistryImpl",
        new DirectServletRegistryImplCBP().withDuplicatePatchingProtection());

    /**
     * Fixes the minifier when rebel.xml is used.
     */
    integration.addIntegrationProcessor(
        "com.liferay.portal.servlet.filters.minifier.MinifierFilter",
        new MinifierFilterCBP().withDuplicatePatchingProtection());

    /**
     * A little bit logging for Liferay file copy
     */
    if (LoggerFactory.getInstance().isTraceEnabled()) {
      integration.addIntegrationProcessor(
          "com.liferay.portal.kernel.util.FileUtil",
          new FileUtilCBP().withDuplicatePatchingProtection());
    }

    /**
     * Disables I18N cache for our portlet refresh.
     */
    integration.addIntegrationProcessor(
        "com.liferay.faces.portal.el.I18N",
        new I18NCBP().withDuplicatePatchingProtection());

    /**
     * Make internal Servlet Context accessible from mojarra and other plugins
     */
    integration.addIntegrationProcessor(
        "com.liferay.faces.bridge.filter.internal.PortletContextBridgeLiferayImpl",
        new PortletContextBridgeLiferayImplCBP().withDuplicatePatchingProtection());

    integration.addIntegrationProcessor(
        new String[] {
            "com.liferay.portlet.PortletContextImpl",
            "com.liferay.portlet.internal.PortletContextImpl"
        },
        new PortletContextImplCBP().withDuplicatePatchingProtection());

    /**
     * SASS recompilation support with rebel paths (only for 6.2+)
     */
    try {
      Class.forName("com.liferay.portal.servlet.filters.aggregate.ServletAggregateContext");

      integration.addIntegrationProcessor(
          "com.liferay.portal.servlet.filters.dynamiccss.DynamicCSSFilter",
          new DynamicCSSFilterCBP().withDuplicatePatchingProtection());
      integration.addIntegrationProcessor(
          "com.liferay.portal.servlet.filters.dynamiccss.DynamicCSSUtil",
          new DynamicCSSUtilCBP().withDuplicatePatchingProtection());

      integration.addIntegrationProcessor(
          "com.liferay.portal.servlet.filters.cache.CacheFilter",
          new CacheFilterCBP().withDuplicatePatchingProtection());
      integration.addIntegrationProcessor(
          "com.liferay.portal.servlet.filters.aggregate.AggregateFilter",
          new AggregateFilterCBP().withDuplicatePatchingProtection());
      integration.addIntegrationProcessor(
          "com.liferay.portal.servlet.filters.header.HeaderFilter",
          new HeaderFilterCBP().withDuplicatePatchingProtection());
    }
    catch (ClassNotFoundException ignored) {
    }

    integration.addIntegrationProcessor(
        "com.liferay.portal.scripting.ruby.RubyExecutor",
        new RubyExecutorCBP().withDuplicatePatchingProtection());

    if (ConfigurationFactory.getInstance().getBoolean("rebel.enable_test_hacks")) {
      integration.addIntegrationProcessor(
          new String[] {
              "com.liferay.portal.kernel.deploy.auto.BaseAutoDeployListener",
              "com.liferay.portal.deploy.auto.PluginAutoDeployListenerHelper", // 7.0
          },
          new BaseAutoDeployListenerCBP().withDuplicatePatchingProtection());
    }

    /**
     * Liferay version reporting
     */
    integration.addIntegrationProcessor(
        new String[] {
            "com.liferay.portal.servlet.MainServlet",
            "com.liferay.portal.internal.servlet.MainServlet"
        },
        new MainServletCBP().withDuplicatePatchingProtection());

    integration.addIntegrationProcessor(
        "com.liferay.portal.kernel.language.UTF8Control",
        new UTF8ControlCBP().withDuplicatePatchingProtection());
    integration.addIntegrationProcessor("com.liferay.portal.servlet.ComboServlet",
        new ComboServletCBP().withDuplicatePatchingProtection());

    /**
     *  Performance optimization
     */
    integration.addIntegrationProcessor(
        "com.liferay.ip.geocoder.internal.IPGeocoderImpl",
        new IPGeocoderImplCBP()
    );

    integration.addIntegrationProcessor(
        "com.liferay.portal.spring.aop.AdvisedSupportAdapter",
        new AdvisedSupportAdapterCBP()
    );

    integration.addIntegrationProcessor(
        "com.liferay.portal.kernel.util.ProxyUtil$DelegateInvocationHandler",
        new DelegateInvocationHandlerCBP());
  }

  public boolean checkDependencies(ClassLoader cl, ClassResourceSource crs) {
    return crs.getClassResource("com.liferay.portal.deploy.hot.HookHotDeployListener") != null;
  }

  public String getId() {
    return "liferay_plugin";
  }

  public String getName() {
    return "Liferay plugin";
  }

  public String getDescription() {
    return "<li>Reloading of JSP hooks</li>" +
        "<li>Reloading of portlet properties and xml configuration files</li>";
  }

  public String getAuthor() {
    return null;
  }

  public String getWebsite() {
    return null;
  }

}
