/**
 * Copyright (C) 2013 ZeroTurnaround OU
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License v2 as published by
 * the Free Software Foundation, with the additional requirement that
 * ZeroTurnaround OU must be prominently attributed in the program.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You can find a copy of GNU General Public License v2 from
 *   http://www.gnu.org/licenses/gpl-2.0.txt
 */
package org.zeroturnaround.jrebel.liferay.cbp;

import org.zeroturnaround.bundled.javassist.CannotCompileException;
import org.zeroturnaround.bundled.javassist.ClassPool;
import org.zeroturnaround.bundled.javassist.CtClass;
import org.zeroturnaround.bundled.javassist.CtMethod;
import org.zeroturnaround.bundled.javassist.CtNewMethod;
import org.zeroturnaround.bundled.javassist.expr.ExprEditor;
import org.zeroturnaround.bundled.javassist.expr.MethodCall;
import org.zeroturnaround.bundled.javassist.expr.NewExpr;
import org.zeroturnaround.javarebel.LoggerFactory;
import org.zeroturnaround.javarebel.RebelServletContext;
import org.zeroturnaround.javarebel.integration.support.CacheAwareJavassistClassBytecodeProcessor;
import org.zeroturnaround.jrebel.liferay.LiferayPlugin;

/**
 * When rebel.xml <code>web</code> parts target dir has the "/html/js" dir inside then JavaScript minifier stops working.
 * e.g. they will scan the dir from the rebel.xml path and the server dir is skipped.
 * 
 * Quick fix is: ?js_fast_load=0 or javascript.fast.load=false
 * 
 * @author Andres Luuk
 */
public class MinifierFilterCBP extends CacheAwareJavassistClassBytecodeProcessor {

  private static final String JR_LOGGER =  "LoggerFactory.getLogger(\"" + LiferayPlugin.PRODUCT_PREFIX + "\")";
  
  public static final String READ_METHOD_WITH_STRING_SIGNATURE = "(Ljava/lang/String;)Ljava/lang/String;";

  public void process(ClassPool cp, ClassLoader cl, CtClass ctClass) throws Exception {
    cp.importPackage("java.io");
    cp.importPackage("org.zeroturnaround.javarebel");

    CtMethod m = ctClass.getDeclaredMethod("getMinifiedBundleContent");

    ctClass.addMethod(CtNewMethod.make("" +
        "public String jrFormat(String fullprefix, String prefix, String original) {" +
        "  try {" +
        "    if (prefix.endsWith(\"/\")) {" +
        "      original = prefix + original.substring(fullprefix.length() + 1);" +
        "    } else {" +
        "      original = prefix + original.substring(fullprefix.length());" +
        "    }" +
        "  } catch (StringIndexOutOfBoundsException e) {" +
        "    " + JR_LOGGER + ".info(\"Problem substringing '{}' with '{}'\",original, fullprefix);" +
        "  }" +
        "  return original;" +
        "}", ctClass));

    m.addLocalVariable("jrFixedPrefix", cp.get(String.class.getName()));
    m.addLocalVariable("jrServletContext", cp.get(RebelServletContext.class.getName()));
    m.addLocalVariable("jrOriginalPath", cp.get(String.class.getName()));
    m.addLocalVariable("jrCacheFileName", cp.get(String.class.getName()));

    m.insertBefore("{" +
        "  jrCacheFileName = \"\"; " +
        "}");

    m.instrument(new ExprEditor() {
      public void edit(NewExpr e) throws CannotCompileException {
        LoggerFactory.getLogger(LiferayPlugin.PRODUCT_PREFIX).info("newxp test: " + e.getClassName() + " " + e.getLineNumber());
        if ("java.io.File".equals(e.getClassName())) {
          e.replace("" +
              "$_ = $proceed($$); " +
              "if ((jrCacheFileName.length() == 0 || !$1.endsWith(jrCacheFileName)) && !$_.exists()) {" + //JR-2405 - skip cache file, not sure if all version have it so use null check just in case
              "  $_ = new File(jrServletContext.getRealPath(jrFormat(jrFixedPrefix, jrOriginalPath, $1)));" +
              "}" +
              "");
        }
      }

      public void edit(MethodCall m) throws CannotCompileException {
        LoggerFactory.getLogger(LiferayPlugin.PRODUCT_PREFIX).info("Method test: " + m.getMethodName() + " " + m.getClassName() + " " + m.getLineNumber());
        if ("getRealPath".equals(m.getMethodName())) {
          m.replace("" +
              "$_ = $proceed($$);" +
              "jrFixedPrefix = $_;" +
              "jrServletContext = (RebelServletContext) $1;" +
              "jrOriginalPath = $2;" +
              "");
        } else if ("read".equals(m.getMethodName())
            && READ_METHOD_WITH_STRING_SIGNATURE.equals(m.getSignature())) { // JR-1229
          m.replace("" +
              "File f = new File($1);" +
              "if (!f.exists()) {" +
              "  $1 = jrServletContext.getRealPath(jrFormat(jrFixedPrefix, jrOriginalPath, $1));" +
              "}" +
              "$_ = $proceed($$);" +
              "");
        } else if ("getCacheFileName".equals(m.getMethodName()) || "toString".equals(m.getMethodName())) { //6.0.6 uses toString
          m.replace("" +
              "$_ = $proceed($$);" +
              "jrCacheFileName = $_;" +
              "");
        }
      }
    });
  }
}
